import shutil
import logging

import pymongo
import requests
import kazoo.client
import kazoo.recipe.barrier
import kazoo.handlers.threading

from sandbox import sdk2
from sandbox.sdk2 import helpers
from sandbox.sdk2.helpers import subprocess as sp

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt

import sandbox.projects.sandbox.resources as sb_resources
import sandbox.projects.sandbox.mongo_suite.consts as consts


ZK_HOSTS_URL = "https://c.yandex-team.ru/api/tag2hosts/sandbox_zk"
ZK_RETRY_ATTEMPTS = 100
ZK_MAX_DELAY = 10


class ServiceMongoShardBackuper(sdk2.ServiceTask):
    """ Backup Sandbox MongoDB replica set """

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 6 * 3600
        shards = sdk2.parameters.List(
            "Shards to backup", description="Expected format is shardname/host:port", required=True
        )
        zk_barrier_path = sdk2.parameters.String("Zookeeper barrier path", required=True, default="mongdump_barrier")
        zk_barrier_num = sdk2.parameters.Integer(
            "Number of clients to enter the barrier to proceed", required=True, default=1
        )
        fsync = sdk2.parameters.Bool("Call db.fsync() before backup", default=False)
        backup_ttl = sdk2.parameters.Integer("For how long should a backup be accessible", default=60)

    def _zk_connect(self):
        def zk_listener(state):
            logging.info("Zookeeper state changed to '%s'", state)
            if state == kazoo.client.KazooState.LOST:
                pass

        zk_hosts = ",".join((
            "{}:2181".format(_.strip())
            for _ in requests.get(ZK_HOSTS_URL).text.split()
        ))
        if not zk_hosts:
            raise common.errors.TaskFailure("No ZK hosts obtained -- cannot synchronize with other tasks")

        handler = kazoo.handlers.threading.SequentialThreadingHandler()
        client = kazoo.client.KazooClient(
            hosts=zk_hosts,
            connection_retry=dict(
                max_tries=ZK_RETRY_ATTEMPTS,
                sleep_func=handler.sleep_func,
                max_delay=10
            ),
            handler=handler,
            randomize_hosts=False
        )
        client.add_listener(zk_listener)
        return client

    @staticmethod
    def _mongoclient(host, port):
        return pymongo.MongoClient("mongodb://{}:{}".format(host, port))

    def backup_shard(self, res_path, shard_name, hp):
        logging.debug("Backing up shard %s, host %s", shard_name, hp)
        shard_backup_path = res_path.joinpath(shard_name)
        shard_backup_directory = res_path.joinpath(shard_name + "_dir")
        shard_backup_directory.mkdir()

        # older versions of mongodump can't resolve records which point to IPv6-only hosts,
        # but why resolve anything at all if the database is known to run on localhost?
        host, port = "localhost", hp.split(":")[1]
        conn = self._mongoclient(host, port)

        # m["name"] has host:port format
        shard_info = filter(lambda m: m["name"] == hp, conn.admin.command(consts.Command.GET_STATUS)["members"])
        is_primary = next(iter(shard_info), {}).get("stateStr") == consts.ShardState.PRIMARY

        # this should not happen, unless there were re-elections during the task's enqueuing, which is unlikely
        assert not is_primary, "Shard {} on {} is primary! Dangerous situation.".format(shard_name, hp)

        if self.Parameters.fsync:
            logging.info("Calling fsync() WITHOUT lock for {}/{}...".format(shard_name, hp))
            conn.fsync()
        else:
            logging.info("Not locking/flushing anything")

        with helpers.ProcessLog(self, logger="mongodump_{}".format(shard_name)) as pl:
            rc_mongodump = sp.Popen(
                [
                    "/usr/bin/nice",
                    "mongodump",
                    "--oplog",
                    "--out", str(shard_backup_directory),
                    "--host", host,
                    "--port", port,
                    "-vvvv",
                ], stdout=pl.stdout, stderr=sp.STDOUT
            ).wait()
            rc_tar = sp.Popen(
                [
                    "tar",
                    "-I", "pixz -p14",
                    "-cf",
                    "{}.tar.pixz".format(str(shard_backup_path)),
                    str(shard_backup_directory),
                ]
            ).wait()
            shutil.rmtree(str(shard_backup_directory))
            return rc_mongodump, rc_tar

    def unlock_shard(self, shard_name, host):
        logging.info("Unlocking shard %s", shard_name)
        host, port = "localhost", host.split(":")[1]
        self._mongoclient(host, port).unlock()
        logging.info("%s unlocked", shard_name)

    def on_prepare(self):
        if self.Parameters.zk_barrier_num < 2:
            return

        zk = self._zk_connect()
        zk.start()
        logging.debug("Waiting at the barrier (%s)...", self.Parameters.zk_barrier_path)
        # noinspection PyTypeChecker
        kazoo.recipe.barrier.DoubleBarrier(
            zk, str(self.Parameters.zk_barrier_path), self.Parameters.zk_barrier_num
        ).enter()
        logging.debug("Go!")

    def on_execute(self):
        siblings = list(ServiceMongoShardBackuper.find(parent=self.parent).limit(0))
        stale_tasks = filter(lambda _: _.status != ctt.Status.EXECUTING, siblings)
        if stale_tasks:
            logging.warning(
                "Not all tasks are executing right now (and they should be): %r",
                [(_.id, _.status) for _ in stale_tasks]
            )

        shards = {
            name: host
            for name, host in (s.split("/") for s in self.Parameters.shards)
        }

        meta = sb_resources.SandboxShardGroupBackup(
            self, "Backup of shards {}".format(self.Parameters.shards),
            str(sdk2.path.Path("backups").joinpath("shards")),
            ttl=self.Parameters.backup_ttl
        )
        if meta.path.exists():
            shutil.rmtree(str(meta.path))
        meta.path.mkdir(parents=True)

        failed = True
        failed_shard, returncodes = (None,) * 2
        # noinspection PyBroadException
        try:
            for shard_name, host in shards.iteritems():
                rc = self.backup_shard(meta.path, shard_name, host)
                if any(rc):
                    failed_shard, returncodes = shard_name, rc
                    break
            failed = False

        except:
            logging.exception("Exception during backup processes")

        finally:
            # noinspection PyBroadException
            try:
                for shard_name, host in shards.iteritems():
                    self.unlock_shard(shard_name, host)

            except:
                logging.exception("Exception during unlock/status restore")

        if failed:
            message = (
                "Failed to backup {} (return codes {})".format(failed_shard, returncodes)
            ) if failed_shard else (
                "Failed to backup a shard, please look for exceptions in debug.log"
            )
            raise common.errors.TaskFailure(message)

        sdk2.ResourceData(meta)
