"""
Parses Sandbox XMLRPC log files on daily basis to provide regular statistics about XMLRPC API usage.
"""

import os
import sys
import logging
import datetime as dt

from sandbox import common
import sandbox.common.types.client as ctc

from sandbox import sandboxsdk


class CustomDateParameter(sandboxsdk.parameters.SandboxStringParameter):
    """ Api log files date to be parsed. """
    description = "Custom date (YYYY-MM-DD) instead of yesterday"
    name = 'custom_date'
    required = False


class TaskTime(object):
    def __init__(self, date):
        self.now = dt.datetime.now().date()
        self.yesterday = dt.timedelta(days=1)
        self.date = dt.datetime.strptime(date, '%Y-%m-%d').date() if date else self.now - self.yesterday


class ParseSandboxApiLogs(sandboxsdk.task.SandboxTask):

    execution_space = 3072
    type = 'PARSE_SANDBOX_API_LOGS'
    client_tags = ctc.Tag.SERVER
    SERVICE = True

    input_parameters = (CustomDateParameter,)

    PRODUCTION_ROOT = '/home/zomb-sandbox'
    PARSER_BINARY = 'scripts/xmlrpc_log_parser.py'

    def on_execute(self):
        task_time = TaskTime(self.ctx.get(CustomDateParameter.name))

        self._exec_stat(task_time, '/var/log/sandbox/xmlrpc.log', 'statistics', 'xmlrpc', 'xmllog.sqlite')
        self._exec_stat(task_time, '/var/log/sandbox/rest.log', 'statistics_rest', 'rest', 'restlog.sqlite')

    def _exec_stat(self, task_time, logname, workdir_path, res_type, sqlite_db_name):
        now, date, yesterday = task_time.now, task_time.date, task_time.yesterday
        if now > date:
            logname += '.{}{}'.format(str(date), '.gz' if now - date > yesterday else '')

        # Determine full path to parser binary
        import yasandbox
        server_root = (
            os.path.join(self.PRODUCTION_ROOT, 'server', 'sandbox')
            if yasandbox.__file__.startswith(self.PRODUCTION_ROOT) else
            os.path.dirname(os.path.dirname(yasandbox.__file__))
        )
        binname = os.path.join(server_root, self.PARSER_BINARY)

        cmd = [
            sys.executable,
            binname,
            '--database', sqlite_db_name,
            '-cl', logname,
            '--user', 'zomb-sandbox',
            'K@search_instrum-sandbox_server'
        ]

        workdir = self.path(workdir_path)
        sandboxsdk.paths.make_folder(workdir)
        channel = sandboxsdk.channel.channel
        res = self.create_resource(self.descr, workdir_path, 'SANDBOX_API_STATISTICS', attributes={'type': res_type})

        with common.fs.WorkDir(workdir):
            # Provide corrent `PYTHONPATH` environment. Temporary, till new core version not released.
            env = os.environ.copy()
            env['PYTHONPATH'] = '/skynet'
            sandboxsdk.process.run_process(cmd, log_prefix="parser_%s" % res_type, shell=True, environment=env)
        channel.sandbox.set_resource_attribute(res.id, 'date', str(date))

        logging.info('Compressing SQLite database.')
        sandboxsdk.process.run_process(['xz', '-1', os.path.join(workdir, sqlite_db_name)],
                                       log_prefix='xz_%s' % res_type)
