"""
Parses Sandbox XMLRPC log files on daily basis to provide regular statistics about XMLRPC API usage.
"""

import os
import sys
import logging
import datetime as dt

from sandbox import common
import sandbox.common.types.client as ctc

from sandbox import sandboxsdk


class CustomDateParameter(sandboxsdk.parameters.SandboxStringParameter):
    """ XMLRPC log files date to be parsed. """
    description = "Custom date (YYYY-MM-DD) instead of yesterday"
    name = 'custom_date'
    required = False


class ParseSandboxXMLRPCLogs(sandboxsdk.task.SandboxTask):

    execution_space = 3072
    type = 'PARSE_SANDBOX_XMLRPC_LOGS'
    client_tags = ctc.Tag.SERVER
    SERVICE = True

    input_parameters = (CustomDateParameter,)
    environment = (sandboxsdk.environments.PipEnvironment('numpy', '1.8.1', use_wheel=True),)

    PRODUCTION_ROOT = '/home/zomb-sandbox'
    PARSER_BINARY = 'scripts/xmlrpc_log_parser.py'
    XMLPRC_LOG_NAME = '/var/log/sandbox/xmlrpc.log'

    def on_execute(self):
        now = dt.datetime.now().date()
        yesterday = dt.timedelta(days=1)
        date = self.ctx.get(CustomDateParameter.name)
        date = dt.datetime.strptime(date, '%Y-%m-%d').date() if date else now - yesterday

        logname = self.XMLPRC_LOG_NAME
        if now > date:
            logname += '.{}{}'.format(str(date), '.gz' if now - date > yesterday else '')

        # Determine full path to parser binary
        import yasandbox
        server_root = (
            os.path.join(self.PRODUCTION_ROOT, 'server', 'sandbox')
            if yasandbox.__file__.startswith(self.PRODUCTION_ROOT) else
            os.path.dirname(os.path.dirname(yasandbox.__file__))
        )
        binname = os.path.join(server_root, self.PARSER_BINARY)

        cmd = [sys.executable, binname, '-cl', logname, 'K@search_instrum-sandbox_server']

        workdir = self.path('statistics')
        sandboxsdk.paths.make_folder(workdir)
        channel = sandboxsdk.channel.channel
        res = self.create_resource(self.descr, 'statistics', 'SANDBOX_XMLRPC_STATISTICS')

        with common.fs.WorkDir(workdir):
            # Provide corrent `PYTHONPATH` environment. Temporary, till new core version not released.
            env = os.environ.copy()
            env['PYTHONPATH'] = '/skynet'
            sandboxsdk.process.run_process(cmd, log_prefix="parser", shell=True, environment=env)
        channel.sandbox.set_resource_attribute(res.id, 'date', str(date))

        logging.info('Compressing SQLite database.')
        sandboxsdk.process.run_process(['xz', '-1', os.path.join(workdir, 'xmllog.sqlite')], log_prefix='xz')
