import time
import types
import inspect
import logging
import textwrap
import datetime as dt
import distutils.version

from sandbox import sdk2
from sandbox import common
import sandbox.common.types.client as ctc

from sandbox.sandboxsdk import environments


def prettify_time(total):
    """
    Make time duration more readable

    :param total: time duration in minutes
    :return: a pretty string (for example, "5 days 1 hour 17 minutes")
    """

    if not total:
        return "now"

    suffixes = ("minute", "hour", "day")
    divisors = (60, 24, 365)
    res = []
    for k in xrange(len(divisors)):
        quantity = total % divisors[k]
        if quantity:
            res += [suffixes[k] + ("s" if quantity > 1 else ""), str(quantity)]
        total /= divisors[k]
    return " ".join(reversed(res))


def walle_url(*hosts):
    return "https://wall-e.yandex-team.ru/?hosts={}".format(",".join(hosts))


class RedeployHosts(sdk2.Task):
    """
    Redeploy a batch of hosts in Wall-E, optionally moving them into another project.
    For Sandbox clients, this also includes a period of downtime, during which
    resources on a client are being replicated. The procedure for each host is as follows:

    1) shut the host down via Sandbox REST API;
    2) (optional) move it to another Wall-E project;
    3) (optional) profile the host;
    4) redeploy it;
    5) wait for a host to become ready.
    """

    WAIT_TIME = (2 * 60 * 60)  # time to wait before each iteration

    encoder = common.api.DateTime()

    class HostState(common.utils.Enum):
        UNPROCESSED = None
        SHUTDOWN = None
        ADDING = None
        CHECKING_DNS = None
        DEPLOYING = None
        FINISHED = None
        FAILED = None

    class Requirements(sdk2.Requirements):
        environments = [environments.PipEnvironment("wall-e.api>=6.1.4,<7.0.0")]

    class Parameters(sdk2.Parameters):
        hosts = sdk2.parameters.String(
            "FQDNs of hosts to redeploy, whitespace-separated",
            multiline=True
        )

        with sdk2.parameters.Group("Maintenance options") as abcdefg:
            cooldown_time = sdk2.parameters.Integer(
                "Time to keep a single host shut down, hours",
                description="This allows Sandbox to replicate resources stored on the host, so that they don't vanish",
                default=8,
            )
            inflight = sdk2.parameters.Integer(
                "How many hosts to keep shut down simultaneously",
                default=30
            )
            inflight_multislots = sdk2.parameters.Integer(
                "How many MULTISLOT hosts to keep shut down simultaneously",
                default=5
            )

        with sdk2.parameters.Group("Wall-E options") as whatever:
            config = sdk2.parameters.String("Deployment config (empty = default for project)")
            do_move = sdk2.parameters.Bool("Move hosts to another project")
            with do_move.value[True]:
                target_project = sdk2.parameters.String("Project name")

        with sdk2.parameters.Group("Wall-E token") as fghjkl:
            vault_item_owner = sdk2.parameters.String("Vault item owner")
            vault_item_name = sdk2.parameters.String("Vault item name")

    class Context(sdk2.Context):
        hosts_states = {}  # FQDN -> (current state, datetime when the host has changed the state)
        native_vlan = None
        allowed_vlans = None
        hosts_steps = {}
        hosts_tags = {}
        shutdown_count = 0
        shutdown_multislot_count = 0

    def __shutdown_checker(self, host_data):
        fqdn, _ = host_data
        to_shutdown = self.Parameters.inflight - self.Context.shutdown_count > 0
        if ctc.Tag.MULTISLOT in self.Context.hosts_tags[fqdn]:
            to_shutdown &= self.Parameters.inflight_multislots - self.Context.shutdown_multislot_count > 0

        return to_shutdown

    def __move_checker(self, host_data):
        if not self.Parameters.do_move:
            return False

        fqdn, (current_state, since) = host_data

        utcnow = dt.datetime.utcnow()
        delta = utcnow - self.encoder.decode(since).replace(tzinfo=None)
        if delta < dt.timedelta(hours=self.Parameters.cooldown_time):
            return False

        host_id = fqdn.split(".")[0]
        info = self.server.client[host_id].read()
        if info["task"] or info["tasks"]:
            logging.debug("Host %s is still busy, skipping", fqdn)
            return False

        info = self.walle_client.get_host(fqdn, fields=("project", "status"))
        if info["status"] != "ready":
            return False
        return True

    @property
    def actions(self):
        return [
            (self.__shutdown_checker, self.__shutdown),
            (self.__move_checker, self.__move),
            (self.Parameters.do_move, self.__check_dns),
            (True, self.__redeploy),
            (True, self.__reenable),
        ]

    @property
    def stale_states(self):
        return {
            "switching-vlans": self.HostState.ADDING,
            "checking-dns": self.HostState.CHECKING_DNS,
            "deploying": self.HostState.DEPLOYING,
        }

    def __coloring(self, state):
        if state == self.HostState.UNPROCESSED:
            return "status_wait_task"
        if state == self.HostState.SHUTDOWN:
            return "status_draft"
        if state in (self.HostState.ADDING, self.HostState.CHECKING_DNS):
            return "status_stable"
        if state == self.HostState.DEPLOYING:
            return "status_executing"
        if state == self.HostState.FINISHED:
            return "status_success"
        return "status_exception"

    @sdk2.header()
    def hosts(self):
        utcnow = dt.datetime.utcnow()

        def host_state(fqdn):
            data = self.Context.hosts_states[fqdn]
            delta = utcnow - self.encoder.decode(data[1]).replace(tzinfo=None)
            return textwrap.dedent("""
                <tr>
                  <td style="padding: 5px">{fqdn}</td>
                  <td style="padding: 5px"><span class="status {coloring}">{state}</span></td>
                  <td style="padding: 5px">{duration}</td>
                  <td style="padding: 5px">
                    <a href="{walle_url}">Wall-E</a> |
                    <a href="{api_url}/client/{host_id}">Sandbox</a>
                  </td>
                </tr>
            """.format(
                fqdn=fqdn,
                state=data[0],
                coloring=self.__coloring(data[0]),
                duration=prettify_time(int(delta.total_seconds() / 60)),
                api_url=common.rest.Client.DEFAULT_BASE_URL,
                walle_url=walle_url(fqdn),
                host_id=fqdn.split(".")[0],
            ))

        fqdns = [
            tuple_[0]
            for tuple_ in sorted(
                self.Context.hosts_states.items(),
                key=lambda t: (t[1][0], distutils.version.LooseVersion(t[0]))
            )
        ]
        return textwrap.dedent("""
            <table>
              <tbody>
                <tr>
                  <th style="padding: 5px">Hostname</th>
                  <th style="padding: 5px">Current status</th>
                  <th style="padding: 5px">Duration</th>
                  <th style="padding: 5px">Links</th>
                </tr>
                {hosts_states}
              </tbody>
            </table>
            <br>
            <br>
            <a href="{walle_url}">All hosts in Wall-E</a>
        """.format(
            hosts_states="\n".join(
                host_state(fqdn)
                for fqdn in fqdns
            ),
            walle_url=walle_url(*fqdns)
        ))

    @common.utils.singleton_property
    def walle_client(self):
        import walle_api

        def wrapper(meth):
            retries = 10
            timeout = 120

            def inner(*a, **kw):
                logging.debug("Calling %s(%r, %r)", meth.__name__, a, kw)
                exc = None
                for _ in xrange(retries):
                    try:
                        return meth(*a, **kw)
                    except walle_api.WalleConnectionError as exc:
                        logging.exception("Caught WalleConnectionError, sleeping for %s", timeout)
                        time.sleep(timeout)

                logging.error(
                    "The call %s(%r, %r) was unsuccessful, re-raising the last exception", meth.__name__, a, kw
                )
                raise exc

            return inner

        access_token = sdk2.Vault.data(self.Parameters.vault_item_owner, self.Parameters.vault_item_name)

        api = walle_api.WalleClient(access_token=access_token)
        for name, method in inspect.getmembers(api, lambda _: isinstance(_, types.MethodType)):
            setattr(api, name, wrapper(method))
        return api

    def __hosts_in_state(self, state):
        return [
            item
            for item in self.Context.hosts_states.items() if
            item[1][0] == state
        ]

    def __save(self, fqdn, state, datetime=None):
        current = self.Context.hosts_states.get(fqdn)
        if current:
            current = current[0]
        logging.debug("%s: %s -> %s", fqdn, current, state)
        self.Context.hosts_states[fqdn] = (state, self.encoder.encode(datetime or dt.datetime.utcnow()))

    def __shutdown(self, fqdn):
        host_id = fqdn.split(".")[0]
        tags = self.server.client[host_id].read()["tags"]
        self.server.client[host_id].tags = list(set(tags) | {str(ctc.Tag.MAINTENANCE)})

        self.Context.shutdown_count += 1
        if ctc.Tag.MULTISLOT in self.Context.hosts_tags[fqdn]:
            self.Context.shutdown_multislot_count += 1

        self.__save(fqdn, self.HostState.SHUTDOWN)

    def __move(self, fqdn):
        info = self.walle_client.get_host(fqdn, fields=("project", "status"))
        if info["project"] == self.Parameters.target_project:
            logging.debug("Host %s is already in desired project -- nothing to do", fqdn)
            return

        self.walle_client.remove_host(
            fqdn, ignore_cms=True, instant=True,
            reason="(task #{}) moving to another project".format(self.id)
        )
        self.walle_client.add_host(fqdn, self.Parameters.target_project, instant=True, with_auto_healing=False)
        self.walle_client.switch_vlans(
            fqdn, vlans=self.Context.allowed_vlans, native_vlan=self.Context.native_vlan
        )
        self.__save(fqdn, self.HostState.ADDING)

    def __check_dns(self, fqdn):
        if self.Parameters.do_move:
            self.walle_client.check_host_dns(
                # skip host's health check so that it doesn't turn dead due to unreachability
                # because of network being configured without project_id consideration
                fqdn, check=False, with_auto_healing=False,
                reason="(task #{}) updating DNS after switching VLANs".format(self.id)
            )
            self.__save(fqdn, self.HostState.CHECKING_DNS)

    def __redeploy(self, fqdn):
        self.walle_client.redeploy_host(
            fqdn, with_auto_healing=False,
            config=self.Parameters.config or None, ignore_maintenance=True, ignore_cms=True,
            reason="(task #{}) redeploying".format(self.id)
        )
        self.__save(fqdn, self.HostState.DEPLOYING)

    def __reenable(self, fqdn):
        host_id = fqdn.split(".")[0]
        tags = self.server.client[host_id].read()["tags"]
        self.server.client[host_id].tags = list(set(tags) - {str(ctc.Tag.MAINTENANCE)})

        self.Context.shutdown_count -= 1
        if ctc.Tag.MULTISLOT in self.Context.hosts_tags[fqdn]:
            self.Context.shutdown_multislot_count -= 1

        self.__save(fqdn, self.HostState.FINISHED)

    def advance_host(self, host_data):
        fqdn, (current_state, since) = host_data

        if current_state not in (self.HostState.UNPROCESSED, self.HostState.SHUTDOWN):
            current_walle_status = self.walle_client.get_host(fqdn)["status"]
            if (
                current_walle_status != "ready" or
                not (current_state == self.HostState.CHECKING_DNS and current_walle_status == "dead")
            ):
                logging.debug("Host %s has failed the process, please fix it manually", fqdn)
                self.__save(fqdn, self.HostState.FAILED)
                return

            for walle_status, state in self.stale_states.iteritems():
                if walle_status == current_walle_status and state == current_state:
                    logging.debug(
                        "The host %s is still in state %s/%s, skipping", fqdn, current_walle_status, current_state
                    )
                    return

        current_step = self.Context.hosts_steps.get(fqdn, 0)
        actions = self.actions
        if current_step >= len(actions):
            logging.debug("Marking %s as finished", fqdn)
            self.__save(fqdn, self.HostState.FINISHED)
            return

        checker, method = actions[current_step]
        logging.debug("Trying step #%d for %s: %s()", current_step, fqdn, method.__name__)
        do_advance = checker(host_data) if callable(checker) else checker
        if do_advance:
            method(fqdn)
            self.Context.hosts_steps[fqdn] = current_step + 1
        else:
            logging.debug("The checker %r() said False for %s, host is not ready yet, skipping", checker, fqdn)

    def on_execute(self):
        with self.memoize_stage.run_once:
            fqdns = self.Parameters.hosts.split()
            for fqdn in fqdns:
                self.__save(fqdn, self.HostState.UNPROCESSED)

            if self.Parameters.target_project:
                info = self.walle_client.get_project(
                    self.Parameters.target_project,
                    fields=("native_vlan", "owned_vlans")
                )
                self.Context.native_vlan = info["native_vlan"]
                self.Context.allowed_vlans = info["owned_vlans"]

            ids = [_.split(".")[0] for _ in fqdns]
            clients_info = self.server.client.read(limit=len(ids), id=",".join(ids))["items"]
            self.Context.hosts_tags = {
                _["fqdn"]: _["tags"]
                for _ in clients_info
            }
            self.Context.save()

        for host_data in self.Context.hosts_states.items():
            self.advance_host(host_data)
            self.Context.save()

        unique_states = set(_[0] for _ in self.Context.hosts_states.values())
        final_states = {self.HostState.FAILED, self.HostState.FINISHED}
        if not unique_states <= final_states:
            raise sdk2.WaitTime(self.WAIT_TIME)

        for state in final_states:
            hosts = [fqdn for fqdn, _ in self.__hosts_in_state(state)]
            logging.info(
                "%d hosts %s the process:\n%s",
                len(hosts), state, "\n".join(hosts)
            )
            self.set_info(
                "<a href='https://wall-e.yandex-team.ru/?hosts={}'>{} hosts in Wall-E</a>".format(
                    ",".join(hosts), state
                ),
                do_escape=False
            )

        if self.HostState.FAILED in unique_states:
            raise common.errors.TaskFailure()
