# -*- coding: utf-8 -*-

import os
import re
import shutil
import logging
import collections
import itertools as it

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.sandboxsdk import copy
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import parameters

from sandbox.projects.common import string
from sandbox.projects.common import utils
from sandbox.projects.common.ya_deploy import release_integration
from sandbox.projects.common.nanny import nanny


class RemoteCopyResource(nanny.ReleaseToNannyTask, release_integration.ReleaseToYaDeployTask, task.SandboxTask):
    """
        DEPRECATED. Use REMOTE_COPY_RESOURCE_2 (sdk2 version, compatible)

        **Описание**
            Создание ресурса Sandbox из файла/торрента (фактически, удалённое копирование в Sandbox).

        **Ресурсы**
            *Необходимые для запуска ресурсы и параметры*
                * Resource type: тип создаваемого ресурса
                * Remote file name: путь до источника ресурса
                * Remote file protocol: выбор протокола копирования
                * Set attrs to resources: назначаемые ресурсу атрибуты (например: attr1=v1, attr2=v2)

            *Создаваемые ресурсы*
                * Создаёт ресурс указанного типа


        **Тонкости вызова через XMLRPC**
        Если вызывать задачу через xmlrpc, то, чтобы явно указать архитектуру загружаемого ресурса,
        необходимо установить ключ контекста, именуемый по шаблону ``{resource_type}_resource_arch``,
        со значением, равным архитектуре загружаемого ресурса. Если устраивает автоматика,
        то для всех ресурсов, у которых запрещено указывать архитектуру
        ``any``, будет выбрана архитектура хоста, на котором происходит выполнение, собственно,
        задачи по загрузке этого ресурса.
        Примечание: архитектура загружаемого ресурса не должна зависеть от архитектуры хоста, на котором,
        собственно, производится эта самая загрузка. Передав параметр "arch" = "linux", будет
        ограничено множество хостов, на котором будет произведена загрузка ресурса,
        но не признак архитектуры загружаемого ресурса.
    """

    type = "REMOTE_COPY_RESOURCE"
    SERVICE = True

    # TODO: SANDBOX-3450 Temporary deny execution on newly layouted hosts.
    client_tags = (ctc.Tag.GENERIC | ctc.Tag.STORAGE) & ~ctc.Tag.NEW_LAYOUT

    class ResName(parameters.SandboxStringParameter):
        name = "created_resource_name"
        description = "Created resource name"
        default_value = "RESOURCE"
        required = True

    class FileName(parameters.SandboxStringParameter):
        name = "remote_file_name"
        description = "Remote file name"
        required = True

    class RsyncCopyLinks(parameters.SandboxBoolParameter):
        name = "rsync_copy_links"
        description = "Rsync transform symlink into referent file/dir"

    class FileProtocol(parameters.SandboxStringParameter):
        name = "remote_file_protocol"
        description = "Remote file protocol"
        choices = [(fp, fp) for fp in ("rcp", "http", "rsync", "svn", "skynet", "scp")]
        required = True
        default_value = "skynet"

    FileProtocol.sub_fields = {
        "rsync": [RsyncCopyLinks.name]
    }

    class DoNotRemove(parameters.SandboxBoolParameter):
        name = "store_forever"
        description = "Store resource forever (assign TTL to infinity)"
        do_not_copy = True
        default_value = False

    class ResAttrs(parameters.SandboxStringParameter):
        name = "resource_attrs"
        description = "Set attrs to resources (e.g.: attr1=v1, attr2=v2)"
        do_not_copy = True

    class IPv4Required(parameters.SandboxBoolParameter):
        name = "ipv4_required"
        description = "Run task on hosts with IPv4"

    input_parameters = utils.get_resource_type_params() + [
        ResName,
        FileName,
        FileProtocol,
        RsyncCopyLinks,
        DoNotRemove,
        ResAttrs,
        IPv4Required,
        # Ya.Deploy release integration: https://wiki.yandex-team.ru/deploy/release-integration/
        release_integration.ReleaseToYaDeployParameter,
        release_integration.YpTokenVaultParameter,
    ]

    @property
    def dns(self):
        return ctm.DnsType.DNS64 if self.ctx.get(self.IPv4Required.name) else ctm.DnsType.DEFAULT

    def arcadia_info(self):
        return None, "Default release subject", None

    # we should create resource before RemoteCopy start
    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        resource = self.create_resource(
            self.descr, self.ResName.default_value,
            self.ctx['resource_type'],
            arch=self.ctx.get('resource_arch', ctm.OSFamily.ANY)
        )
        if self.ctx.get('deferred_upload'):
            from sandbox.yasandbox import manager
            manager.task_manager.register_dep_resource(self.id, resource.id)
            resource = manager.resource_manager.load(resource.id)
            resource.mark_broken()
        self.ctx['result_resource_id'] = resource.id

    def postprocess_upload(self):
        # This method should be executed on a host, which performed an actual data upload.
        common_prefix = None
        basedir = self.ctx['deferred_upload']
        logging.info("Postprocessing HTTP upload stored at '%s'.", basedir)
        counters = collections.Counter()
        for root, dirs, files in os.walk(basedir):
            for f in files:
                f = os.path.join(root, f)
                counters["size"] += os.stat(f).st_size
            counters["files"] += len(files)
            counters["dirs"] += len(dirs)

        for fname in os.listdir(basedir):
            shutil.move(os.path.join(basedir, fname), self.abs_path(fname))
            common_prefix = (
                ''.join(c[0] for c in it.takewhile(lambda x: all(x[0] == y for y in x), it.izip(common_prefix, fname)))
                if common_prefix else
                fname
            )
        shutil.rmtree(basedir, ignore_errors=True)
        logging.info(
            "Resource's common prefixed detected as '%s', %d directories, %d files of total size %s.",
            common_prefix, counters["dirs"], counters["files"], common.utils.size2str(counters["size"])
        )
        if not common_prefix or not os.path.exists(os.path.join(self.abs_path(common_prefix))):
            raise errors.SandboxTaskFailureError("Uploaded files has no any common directory prefix.")
        self.change_resource_basename(self.ctx['result_resource_id'], common_prefix)
        if not counters["size"]:
            raise errors.SandboxTaskFailureError("Data of zero length has been uploaded.")

    @staticmethod
    def _remote_copy(remote_file_name, dest_path, log_dir, tout, **kws):
        r = copy.RemoteCopy(remote_file_name, dest_path, log_dir=log_dir)
        logging.info("Using %r to fetch remote data.", r)
        params = {}
        if not isinstance(r, copy.RemoteCopySvn):
            params["timeout"] = int(tout) / 2 if tout else None
        params.update(kws)
        r(**params)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        if self.ctx.get(release_integration.ReleaseToYaDeployParameter.name):
            release_integration.ReleaseToYaDeployTask.on_release(self, additional_parameters)
        task.SandboxTask.on_release(self, additional_parameters)

    def on_execute(self):
        import api.copier.errors

        resource_attributes = self.ctx.get(self.ResAttrs.name)
        if resource_attributes:
            logging.info('Set resource attributes %s', resource_attributes)
            for k, v in string.parse_attrs(resource_attributes).iteritems():
                channel.channel.sandbox.set_resource_attribute(self.ctx["result_resource_id"], k, v)

        if self.ctx.get('deferred_upload'):
            return self.postprocess_upload()

        unstable = []
        exclude = []

        if 'unstable' in self.ctx:
            unstable.extend(self.ctx['unstable'])
        if 'exclude' in self.ctx:
            exclude.extend(self.ctx['exclude'])

        if str(self.ctx['resource_type']) == "SEARCH_DATABASE":
            exclude.extend([
                'oldindexarc',
                'oldindexarr',
                'oldindexdir',
                'oldindexiarr',
                'oldindexinv',
                'oldindexkey',
            ])

        logging.info('Set resource file name')
        resource_specified_name = self.ctx.get(self.ResName.name)
        if resource_specified_name:
            self.change_resource_basename(self.ctx['result_resource_id'], resource_specified_name)

        logging.info('Get resource object')
        resource = self._read_resource(self.ctx['result_resource_id'], sync=False)

        # при копировании скайнетом нужно сначала скопировать
        # во временную директорию
        # а потом уже перенести в ресурс
        if (
            self.ctx[self.FileProtocol.name] == "skynet" and
            self.ctx[self.FileName.name].startswith("rbtorrent:")
        ):

            dest_path = os.path.join(
                os.path.dirname(resource.abs_path()), '_fetched_resource'
            )
        else:
            dest_path = resource.abs_path()
        try:
            tout = self.ctx.get('kill_timeout')
            remote_file_name = self.ctx[self.FileName.name]
            remote_file_protocol = self.ctx[self.FileProtocol.name]

            kws = {}
            if remote_file_protocol == 'rsync':
                match = re.match(r'(.+)::(.+)', remote_file_name)
                if match:
                    remote_file_name = 'rsync://{}/{}'.format(match.group(1), match.group(2))
                if self.ctx[self.RsyncCopyLinks.name]:
                    kws = {'copy-links': True}
            elif remote_file_protocol == 'skynet':
                kws = {
                    'unstable': unstable,
                    'exclude': exclude,
                    'fallback_to_bb': True,
                }

            self._remote_copy(
                remote_file_name, dest_path, self.log_path(), tout,
                **kws
            )
        except api.copier.errors.ResourceNotAvailable as ex:
            raise errors.SandboxTaskFailureError(ex)

        # переносим файл или директорию в ресурс
        if (
            self.ctx[self.FileProtocol.name] == "skynet" and
            self.ctx[self.FileName.name].startswith("rbtorrent:")
        ):
            copied_files = os.listdir(dest_path)
            if not copied_files:
                raise errors.SandboxTaskFailureError(
                    "Error while copying from SkyNet: {}".format(copied_files)
                )
            if len(copied_files) > 1:
                shutil.move(dest_path, os.path.join(os.path.dirname(dest_path), resource_specified_name))
            else:
                res_file = os.path.join(dest_path, copied_files[0])
                if os.path.isdir(res_file):
                    os.rename(res_file, resource.abs_path())
                else:
                    shutil.move(res_file, resource.abs_path())
                shutil.rmtree(dest_path)
        if self.ctx.get(self.DoNotRemove.name, False):
            channel.channel.sandbox.set_resource_attribute(resource.id, "ttl", "inf")

        # set a+x bit to executable resource
        # Jira: SANDBOX-1245
        if resource.type.executable:
            logging.info("Set 0555 bit to executable resource")
            paths.chmod(resource.abs_path(), 0o555, recursively=False)
