# -*- coding: utf-8 -*-
import six

import os
import re
import shutil
import logging

from sandbox import sdk2
from sandbox.sdk2 import copy
from sandbox.sdk2 import paths

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.common import errors

from sandbox.projects.common import binary_task
from sandbox.projects.common import string
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.ya_deploy import release_integration
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import task_env


class RemoteCopyResource2(
    binary_task.LastBinaryTaskRelease,
    sdk2.ServiceTask,
    nanny.ReleaseToNannyTask2,
    release_integration.ReleaseToYaDeployTask2,
):
    """
        **Описание**
            Создание ресурса Sandbox из файла/торрента (фактически, удалённое копирование в Sandbox).

        **Ресурсы**
            *Необходимые для запуска ресурсы и параметры*
                * Resource type: тип создаваемого ресурса
                * Remote file name: путь до источника ресурса
                * Remote file protocol: выбор протокола копирования
                * Set attrs to resources: назначаемые ресурсу атрибуты (например: attr1=v1, attr2=v2)

            *Создаваемые ресурсы*
                * Создаёт ресурс указанного типа


        **Тонкости вызова через XMLRPC**
        Если вызывать задачу через xmlrpc, то, чтобы явно указать архитектуру загружаемого ресурса,
        необходимо установить ключ контекста, именуемый по шаблону ``{resource_type}_resource_arch``,
        со значением, равным архитектуре загружаемого ресурса. Если устраивает автоматика,
        то для всех ресурсов, у которых запрещено указывать архитектуру
        ``any``, будет выбрана архитектура хоста, на котором происходит выполнение, собственно,
        задачи по загрузке этого ресурса.
        Примечание: архитектура загружаемого ресурса не должна зависеть от архитектуры хоста, на котором,
        собственно, производится эта самая загрузка. Передав параметр "arch" = "linux", будет
        ограничено множество хостов, на котором будет произведена загрузка ресурса,
        но не признак архитектуры загружаемого ресурса.
    """

    class Requirements(task_env.TinyRequirements):
        # TODO: SANDBOX-3450 Temporary deny execution on newly layouted hosts.
        client_tags = (ctc.Tag.GENERIC | ctc.Tag.STORAGE) & ~ctc.Tag.NEW_LAYOUT
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Parameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)

        resource_arch = sdk2.parameters.String(
            "Uploaded resource arch",
            choices=[(x, x) for x in ctm.OSFamily],
            default_value="any",
        )
        resource_type = sdk2.parameters.String(
            "Resource type",
            choices=sorted([(rt.name, rt.name) for rt in list(sdk2.resource.AbstractResource)]),
            required=True,
        )
        created_resource_name = sdk2.parameters.String(
            "Created resource name",
            required=True,
            default_value="RESOURCE",
        )
        remote_file_name = sdk2.parameters.String("Remote file name", required=True)
        remote_file_protocol = sdk2.parameters.String(
            "Remote file protocol",
            choices=[(fp, fp) for fp in ("rcp", "http", "rsync", "svn", "skynet", "scp")],
            default_value="skynet",
        )
        with remote_file_protocol.value["rsync"]:
            rsync_copy_links = sdk2.parameters.Bool("Transform symlink into referent file/dir")

        store_forever = sdk2.parameters.Bool(
            "Store resource forever (assign TTL to infinity)",
            do_not_copy=True,
            default_value=False,
        )
        resource_attrs = sdk2.parameters.String(
            "Set attrs to resources (e.g.: attr1=v1, attr2=v2)",
            do_not_copy=True,
        )
        system_resource_attrs = sdk2.parameters.String(
            "Set internal sandbox attrs to resources (e.g.: attr1=v1, attr2=v2)",
            do_not_copy=True,
        )
        ipv4_required = sdk2.parameters.Bool("Run task on hosts with IPv4")

        # These two are skynet-only
        with remote_file_protocol.value["skynet"]:
            exclude = sdk2.parameters.String('Exclude [passed to skynet copier as `exclude` argument]')
            unstable = sdk2.parameters.String('Unstable [passed to skynet copier as `unstable` argument]')

        # Ya.Deploy release integration: https://wiki.yandex-team.ru/deploy/release-integration/
        yp_token_vault = release_integration.YpTokenVaultParameter2()
        release_to_ya_deploy = release_integration.ReleaseToYaDeployParameter2()

        with sdk2.parameters.Output():
            result_resource_id = sdk2.parameters.Resource("Output resource")

    def arcadia_info(self):
        return None, "Default release subject", None

    # we should create resource before RemoteCopy start
    def on_enqueue(self):
        super(RemoteCopyResource2, self).on_enqueue()
        arch = self.Parameters.resource_arch or ctm.OSFamily.ANY
        system_resource_attributes = self.Parameters.system_resource_attrs
        system_attributes = {}
        if system_resource_attributes:
            for k, v in string.parse_attrs(system_resource_attributes).iteritems():
                system_attributes[k] = v
        resource = sdk2.Resource[self.Parameters.resource_type](
            self,
            self.Parameters.description,
            self.Parameters.created_resource_name,
            arch=arch, system_attributes=system_attributes
        )
        self.Parameters.result_resource_id = resource.id

    @staticmethod
    def _remote_copy(remote_file_name, dest_path, log_dir, tout, **kws):
        r = copy.RemoteCopy(remote_file_name, dest_path, log_dir=log_dir)
        logging.info("Using %r to fetch remote data.", r)
        params = {}
        if not isinstance(r, copy.RemoteCopySvn):
            params["timeout"] = int(tout) / 2 if tout else None
        params.update(kws)
        r(**params)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        if self.Parameters.release_to_ya_deploy:
            release_integration.ReleaseToYaDeployTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)

    def _is_skynet(self):
        return (
            self.Parameters.remote_file_protocol == "skynet"
            and self.Parameters.remote_file_name.startswith("rbtorrent:")
        )

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        import api.copier.errors

        resource = sdk2.Resource[self.Parameters.result_resource_id]
        resource_attributes = self.Parameters.resource_attrs
        if resource_attributes:
            logging.info('Set resource attributes %s', resource_attributes)
            for k, v in six.iteritems(string.parse_attrs(resource_attributes)):
                setattr(resource, k, v)

        # при копировании скайнетом нужно сначала скопировать
        # во временную директорию, а потом уже перенести в ресурс
        if self._is_skynet():
            dest_path = str(resource.path.parent / '_fetched_resource')
        else:
            dest_path = str(resource.path)

        try:
            tout = self.Parameters.kill_timeout
            remote_file_name = self.Parameters.remote_file_name
            remote_file_protocol = self.Parameters.remote_file_protocol

            kws = {}
            if remote_file_protocol == 'rsync':
                match = re.match(r'(.+)::(.+)', remote_file_name)
                if match:
                    remote_file_name = 'rsync://{}/{}'.format(match.group(1), match.group(2))
                if self.Parameters.rsync_copy_links:
                    kws = {'copy-links': True}
            elif remote_file_protocol == 'skynet':
                unstable = []
                exclude = []

                if self.Parameters.unstable:
                    unstable.extend(self.Parameters.unstable)
                if self.Parameters.exclude:
                    exclude.extend(self.Parameters.exclude)

                kws = {
                    'unstable': unstable,
                    'exclude': exclude,
                    'fallback_to_bb': True,
                }

            self._remote_copy(
                remote_file_name, dest_path, self.log_path(), tout,
                **kws
            )
        except api.copier.errors.ResourceNotAvailable as ex:
            raise errors.TaskError(ex)

        # move file/directory to resource
        if self._is_skynet():
            copied_files = os.listdir(dest_path)
            eh.ensure(copied_files, "Error while copying from SkyNet: {}".format(copied_files))

            if len(copied_files) > 1:
                logging.info('Move %s to %s', dest_path, resource.path)
                shutil.move(dest_path, str(resource.path))
            else:
                res_file = os.path.join(dest_path, copied_files[0])
                if os.path.isdir(res_file):
                    os.rename(res_file, str(resource.path))
                else:
                    shutil.move(res_file, str(resource.path))
                shutil.rmtree(dest_path)

        if self.Parameters.store_forever:
            resource.ttl = 'inf'

        # set a+x bit to executable resource
        # SANDBOX-1245
        if resource.type.executable:
            logging.info("Set 0o555 bit to executable resource")
            paths.chmod(str(resource.path), 0o555, recursively=False)
