import os
import random
import logging
import requests

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.notification as ctn

from sandbox import sdk2
from sandbox import sandboxsdk

import sandbox.projects.sandbox.resources as sb_resources
from sandbox.projects.sandbox import test_task
from sandbox.projects.sandbox import test_task_2
from sandbox.projects.sandbox import test_mds_upload
from sandbox.projects.sandbox import deploy_binary_task
from sandbox.projects.sandbox import sandbox_acceptance_sdk1
from sandbox.projects.sandbox import sandbox_lxc_image_acceptance

import sandbox.projects.sandbox_ci.sandbox_ci_web4_priemka as serp_priemka


class TaskboxVersions(sdk2.parameters.List):

    default_type = sdk2.parameters.String

    @common.utils.classproperty
    def default_value(cls):
        try:
            from sandbox import taskbox
            return map(str, range(taskbox.REQUIRED_AGE, taskbox.AGE + 1))
        except ImportError:
            return []


ARC_TOKEN_NAME = "arc-token"
ARC_TEST_BUILD_URL = "arcadia:/arc/trunk/arcadia/util"


class SandboxAcceptance(sdk2.Task):
    """
    Task to run sandbox test tasks.
    https://st.yandex-team.ru/SANDBOX-2756
    """

    class Parameters(sdk2.Parameters):
        description = "Sandbox acceptance"

        multislot_task_count = sdk2.parameters.Integer(
            "Number of tasks to run on multislot clients", default=7,
        )
        multislot_porto_task_count = sdk2.parameters.Integer(
            "Number of tasks to run on multislot porto clients", default=7,
        )
        owner_for_serp_acceptance = sdk2.parameters.String(
            "Owner for SERP acceptance task",
            default="SANDBOX_CI_SEARCH_INTERFACES",
            description=(
                "Why? Because there are too many entities tailored to default owner of Sandbox CI tasks "
                "(for example, semaphores or vault records)"
            )
        )
        taskbox_versions = TaskboxVersions("Binaries' versions to test (sandbox.taskbox.AGE).")

        exclude_sandbox_ci_web4_priemka = sdk2.parameters.Bool("Don't run SANDBOX_CI_WEB4_PRIEMKA")

        arc_secret = sdk2.parameters.YavSecret(
            "Yav secret with OAuth token (the key should be '{}')".format(ARC_TOKEN_NAME),
            required=True,
            default="sec-01f5jyr88xdr3z58tjqrnjd1j0"
        )

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        disk_space = 100

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Context):
        local_resource_id = None
        child_tasks_ids = []

    _REST_REQUEST_MAX_LIMIT = 3000

    def _get_available_platforms(self):
        resp = self.server.client.read(limit=self._REST_REQUEST_MAX_LIMIT)
        platforms = list(set(client["platform"] for client in resp["items"]))
        logging.info("Available platforms: %s", platforms)
        return platforms

    def _get_available_container_resource_ids(self):
        settings = common.config.Registry()
        response = self.server.resource.read(
            type=sb_resources.LXC_CONTAINER.name,
            state=ctr.State.READY,
            attrs={
                "released": settings.server.services.packages_updater.release_status,
            },
            limit=self._REST_REQUEST_MAX_LIMIT,
            order="-id"
        )
        containers = {}
        for resource in response["items"]:
            platform = resource["attributes"].get("platform")
            if platform and platform not in containers and platform != ctc.ContainerPlatforms.LUCID:
                containers[platform] = resource["id"]
        logging.info("Available container resources: %s", containers)
        return containers.items()

    @common.utils.singleton_property
    def _notifications(self):
        return self.server.task[self.id].read()["notifications"]

    def _create_sdk1_subtask(self, task_type, description, parameters, requirements=None, kill_timeout=None):
        data = {
            "type": task_type,
            "context": parameters,
            "children": True,
            "description": description,
            "notifications": self._notifications,
            "owner": self.owner,
            "priority": {
                "class": self.Parameters.priority.cls,
                "subclass": self.Parameters.priority.scls,
            },
        }
        if requirements:
            data["requirements"] = requirements
        if kill_timeout:
            data["kill_timeout"] = kill_timeout
        task = self.server.task(data)
        self.server.batch.tasks.start.update([task["id"]])
        return task["id"]

    def _create_yabs_tasks(self, description):
        from sandbox.projects import YabsDebuilder
        yield self._create_sdk1_subtask(
            YabsDebuilder.YabsDebuilder.type,
            description,
            parameters={
                YabsDebuilder.ArcadiaPath.name:
                    "svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/yabs/tests/test-devservices",
                YabsDebuilder.ArcadiaRevision.name: 2025586,
                YabsDebuilder.DryRun.name: True,
            },
        )

    def _create_local_resources(self):
        res = sb_resources.TestTaskResource(self, "test resource", "resource_file")
        res_data = sdk2.ResourceData(res)
        res_data.path.write_text(res.description)
        res_data.ready()
        res2 = test_task_2.TestTask2Resource(self, "test resource 2", "resource_file2")
        res_data2 = sdk2.ResourceData(res2)
        res_data2.path.write_text(res2.description)
        res_data2.ready()
        return res, res2

    def _create_autocheck_tasks(self, description):
        import sandbox.projects.autocheck.AutocheckAcceptance

        arcadia_url = sdk2.svn.Arcadia.freeze_url_revision(sdk2.svn.Arcadia.trunk_url())

        for client_tags in (ctc.Tag.GENERIC, ctc.Tag.PORTOD):
            acceptance_task = sandbox.projects.autocheck.AutocheckAcceptance.AutocheckAcceptance(
                self,
                description=description + " ({})".format(client_tags),
                arcadia_url=arcadia_url,
                check_precommit=False,
                check_postcommit=False,
                check_resources=True,
                client_tags=client_tags,
                arc_token="sec-01fbmgvxk3rszdpvkmn3p15aqc#ci.token",
                sandbox_token="sec-01fbmgvxk3rszdpvkmn3p15aqc#ci.token",
            )
            yield acceptance_task.enqueue().id

        from sandbox.projects import YaPackageAcceptance
        yield self._create_sdk1_subtask(
            YaPackageAcceptance.YaPackageAcceptanceTask.type,
            description,
            parameters={
                YaPackageAcceptance.DontPublishDebianPackagesParameter.name: True,
                "fail_on_any_error": True,
            },
            kill_timeout=10 * 60,
        )

    def _create_test_tasks(self, description, res_id):
        tt_cls = test_task.TestTask
        common_test_task_params = {
            tt_cls.CheckSsh.name: True,
            tt_cls.VaultItemName.name: "sandbox_robot_ssh",
            tt_cls.PipEnvironments.name: "requests,yandex-yt,anyconfig",
            tt_cls.DependentResource.name: res_id,
        }
        for platform in self._get_available_platforms():
            input_params = {
                tt_cls.RamdriveSize.name: int("linux" in platform),
                tt_cls.CheckCoredump.name: "linux" in platform,
                tt_cls.CheckPbcopy.name: "osx" in platform,
            }
            input_params.update(common_test_task_params)
            yield self._create_sdk1_subtask(
                tt_cls.type,
                "{}: platform={}".format(description, platform),
                parameters=input_params,
                requirements={"platform": platform},
                kill_timeout=10 * 60,
            )

        for platform, resource_id in self._get_available_container_resource_ids():
            for privileged in (True, False):
                full_description = "{}: LXC={}, privileged={}".format(description, platform, privileged)
                input_params = {
                    tt_cls.Container.name: resource_id,
                    tt_cls.Privileged.name: privileged,
                    tt_cls.RamdriveSize.name: int("linux" in platform),
                }
                input_params.update(common_test_task_params)
                yield self._create_sdk1_subtask(
                    tt_cls.type,
                    full_description,
                    parameters=input_params,
                    kill_timeout=10 * 60,
                )

                image_acceptance = sandbox_lxc_image_acceptance.SandboxLxcImageAcceptance(
                    self,
                    description=full_description,
                    lxc_resource=resource_id,
                    privileged=privileged,
                    kill_timeout=10 * 60,
                )
                yield image_acceptance.save().enqueue().id

    def test_task_2_common_kwargs(self, test_mounting=False, use_arc=False):
        tt2_cls = test_task_2.TestTask2
        kwargs = {
            "owner": self.Parameters.owner,
            tt2_cls.Parameters.check_vault.name: True,
            tt2_cls.Parameters.vault_item_name.name: "test_data",
            "kill_timeout": 15 * 60,
            "use_arc": use_arc
        }
        if test_mounting:
            kwargs[tt2_cls.Parameters.mount_overlay.name] = True
        if use_arc:
            kwargs.update({
                "arc_secret": str(self.Parameters.arc_secret),
                "build_with_arcadiasdk": True,
                "test_get_arcadia_url": ARC_TEST_BUILD_URL
            })
        return kwargs

    def _create_tasks_for_multislot_client(self, description):
        for _ in xrange(self.Parameters.multislot_task_count):
            task = test_task_2.TestTask2(
                self,
                description="{}: use no caches".format(description),
                check_coredump=True,
                use_cgroup=True,
                **self.test_task_2_common_kwargs(test_mounting=True, use_arc=True)
            )
            task.Requirements.client_tags &= ctc.Tag.Group.LINUX
            task.Requirements.Caches.qwer = None
            del task.Requirements.Caches.qwer

            yield task.save().enqueue().id

    def _create_tasks_for_multislot_porto_client(self, description):
        for _ in xrange(self.Parameters.multislot_porto_task_count):
            task = test_task_2.TestTask2(
                self,
                description="{}: use no caches (PORTOD)".format(description),
                check_coredump=False,
                use_cgroup=False,
                **self.test_task_2_common_kwargs()
            )
            task.Requirements.client_tags = ctc.Tag.PORTOD
            task.Requirements.Caches.qwer = None
            del task.Requirements.Caches.qwer

            yield task.save().enqueue().id

    @common.patterns.singleton_property
    def python3_binary(self):
        return sdk2.service_resources.SandboxTasksBinary.find(
            state=ctr.State.READY,
            attrs={"postcommit_check": "True", "python": 3},
            owner="SANDBOX"
        ).first()

    @common.patterns.singleton_property
    def tasklet_acceptance_binary(self):
        return sdk2.service_resources.SandboxTasksBinary.find(
            state=ctr.State.READY,
            attrs={"task_type": "TASKLET_ACCEPTANCE", "released": "stable"},
            owner="SANDBOX"
        ).first()

    @staticmethod
    def __multislot_requirements(tasks_resource_id=None):
        return {
            "tasks_resource": tasks_resource_id,
            "cores": 1,
            "ram": 1 << 10,  # 1 GiB
        }

    def _create_privileged_test_tasks_2(self, description, res_id):
        yield test_task_2.TestTask2(
            self,
            description="{}: privileged".format(description),
            dependent_resource=res_id,
            privileged=True,
            use_cgroup=True,
            **self.test_task_2_common_kwargs(test_mounting=True, use_arc=True)
        ).enqueue().id
        yield test_task_2.TestTask2(
            self,
            description="{}: python3 privileged".format(description),
            __requirements__={"tasks_resource": self.python3_binary},
            privileged=True,
            use_cgroup=True,
            create_sub_task=True,
            pass_resources_to_subtasks=True,
            sleep_in_subprocess=True,
            create_resource_on_enqueue=True,
            modify_resource=True,
            create_ramdir_on_prepare=True,
            create_ramdir_on_execute=True,
            check_ssh_agent=True,
            sync_resource_in_subprocess=True,
            **self.test_task_2_common_kwargs(test_mounting=True, use_arc=True)
        ).enqueue().id

    def _create_tasklet_acceptance(self, description):
        task_id = self.server.task.create(
            type="TASKLET_ACCEPTANCE", owner=self.owner, description=description,
            requirements={"tasks_resource": self.tasklet_acceptance_binary.id},
            children=True
        )["id"]
        start_result = self.server.batch.tasks.start.update([task_id])[0]
        status, message = map(start_result.get, ("status", "message"))
        if status == ctm.BatchResultStatus.ERROR:
            raise common.errors.TaskError("Can't run subtask {}: {}".format(task_id, message))
        yield task_id

    def _create_test_tasks_2_without_unmount(self, description):
        squashfs_image_res = sdk2.service_resources.SandboxTasksImage.find(state=ctr.State.READY).first()
        for privileged in (False, True):
            yield test_task_2.TestTask2(
                self,
                description="{}: without unmount{}".format(description, (" (privileged)" if privileged else "")),
                privileged=privileged,
                use_cgroup=True,
                mount_image=squashfs_image_res,
                do_not_unmount=True,
                __requirements__=dict(client_tags=ctc.Tag.LXC),
                **self.test_task_2_common_kwargs(True)
            ).enqueue().id

    def _create_task_for_nonlxc_client(self, description):
        client_tags = ~ctc.Tag.NEW_LAYOUT & ~ctc.Tag.LXC & (ctc.Tag.SERVER | ctc.Tag.STORAGE)
        yield test_task_2.TestTask2(
            self,
            description="{}: client_tags={}".format(description, client_tags),
            overwrite_client_tags=True,
            client_tags=client_tags,
            check_coredump=True,
            use_cgroup=True,
            **self.test_task_2_common_kwargs()
        ).enqueue().id

    def _create_test_tasks_with_custom_tasks_archive(self, description):
        res = sdk2.service_resources.SandboxTasksArchive.find(
            state=ctr.State.READY,
            attrs={"auto_deploy": True},
            owner=common.config.Registry().common.service_group,
        ).first()
        task = test_task_2.TestTask2(
            self,
            description="{}: tasks archive".format(description),
            __requirements__=self.__multislot_requirements(res.id),
            **self.test_task_2_common_kwargs()
        )
        task.Requirements.Caches.qwer = None
        del task.Requirements.Caches.qwer
        yield task.save().enqueue().id

    def _create_taskboxed_test_tasks(self, description):
        not_found_ages = []
        for tb_version in self.Parameters.taskbox_versions:
            res = sdk2.service_resources.SandboxTasksBinary.find(
                state=ctr.State.READY,
                owner=common.config.Registry().common.service_group,
                attrs={
                    "released": ctt.ReleaseStatus.STABLE,
                    "target": "sandbox/projects/sandbox",
                    ctr.BinaryAttributes.TASKBOX_ENABLED: "True",
                    ctr.BinaryAttributes.BINARY_AGE: tb_version,
                },
            ).first()
            if res:
                task = test_task_2.TestTask2(
                    self,
                    description="{}: taskboxed task".format(description),
                    overwrite_client_tags=True,
                    client_tags=ctc.Tag.LINUX_XENIAL,
                    __requirements__=self.__multislot_requirements(res.id),
                    **self.test_task_2_common_kwargs()
                )
                task.Requirements.Caches.qwer = None
                del task.Requirements.Caches.qwer
                yield task.save().enqueue().id
            else:
                not_found_ages.append(tb_version)
        if not_found_ages:
            self.set_info(
                "Warning! There are no binary tasks with followed ages: {}".format(", ".join(map(repr, not_found_ages)))
            )

    def _create_deploy_binary_task(self, description):
        yield deploy_binary_task.DeployBinaryTask(
            self,
            description=description,
            arcadia_url="arcadia:/arc/trunk/arcadia",
            target="sandbox/projects/sandbox/deploy_binary_task/bin",
            check_types="DEPLOY_BINARY_TASK",
            integrational_check=False,
            keep_binary_on_fail=False,
            use_arc=True,
            arc_oauth_token="sec-01f5jyr88xdr3z58tjqrnjd1j0#arc-token",
            build_system="semi_distbuild",
            ramdrive_for_cache=True,
            use_yt_cache=True,
            yt_token_vault="yt-token",
            release_ttl=3
        ).enqueue().id

    def _create_serp_test_task(self, description):
        if self.Parameters.exclude_sandbox_ci_web4_priemka:
            return

        if self.Parameters.owner_for_serp_acceptance:
            owner = self.Parameters.owner_for_serp_acceptance
        else:
            owner = self.Parameters.owner

        yield serp_priemka.SandboxCiWeb4Priemka(
            self,
            description=description,
            owner=owner,
            notifications=sdk2.Notification(
                statuses=tuple(ctt.Status.Group.BREAK) + (ctt.Status.FAILURE,),
                recipients=["infraduty-sandbox-priemka"],
                transport=ctn.Transport.EMAIL
            ),
            project_git_base_ref="dev",
            project_github_owner="serp",
            project_github_repo="web4",
            project_build_context="dev",
            reuse_task_cache=True,
            report_github_statuses=False,
            gemini_platforms=["desktop"],
            hermione_platforms=["desktop"],
            pulse_platforms=["desktop"],
            gemini_custom_opts="--grep /index$/",
            hermione_custom_opts="--grep foo",
            palmsync_validate_custom_opts="--skip scenarios --skip tests",
            palmsync_synchronize_custom_opts="--dry",
            arc_ref="trunk",
        ).enqueue().id

    def _create_acceptance_tasks(self, description):
        yield self._create_sdk1_subtask(
            sandbox_acceptance_sdk1.SandboxAcceptanceSDK1.type,
            description,
            parameters={sandbox_acceptance_sdk1.CreateSvnTag.name: True},
            kill_timeout=5 * 60,
        )

    def _test_resource_download(self, resource):
        for i in xrange(4):
            try:
                assert requests.get(resource.http_proxy).text == resource.description
                break
            except requests.RequestException as exc:
                logging.error("Failed to download resource via HTTP (attempt %d): %s", i, exc)
        else:
            raise exc

        rsync_url = random.choice(self.server.resource[resource.id].data.rsync[:])["url"]
        sandboxsdk.copy.RemoteCopyRsync(rsync_url, ".")
        assert open(os.path.basename(str(resource.path)), "r").read() == resource.description

    def _create_test_task_to_release(self, description):
        task_id = test_task_2.TestTask2(
            self,
            description="{}: to release".format(description),
            owner=self.Parameters.owner,
            check_ssh_agent=True,
        ).enqueue().id
        self.Context.tasks_to_release = [task_id]
        yield task_id
        task_id = sdk2.Task[test_task.TestTask.type](
            self,
            description="{}: to release".format(description),
            owner=self.Parameters.owner,
        ).enqueue().id
        self.Context.tasks_to_release.append(task_id)
        yield task_id
        task_id = test_task_2.TestTask2(
            self,
            description="{}: portod to release".format(description),
            owner=self.Parameters.owner,
            check_coredump=True,
            __requirements__={"client_tags": ctc.Tag.PORTOD}
        ).enqueue().id
        self.Context.tasks_to_release.append(task_id)
        yield task_id
        task_id = test_task_2.TestTask2(
            self,
            description="{}: Privileged portod to release".format(description),
            owner=self.Parameters.owner,
            check_coredump=True,
            privileged=True,
            __requirements__={"client_tags": ctc.Tag.PORTOD}
        ).enqueue().id
        self.Context.tasks_to_release.append(task_id)
        yield task_id
        tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
            state=ctr.State.READY,
            attrs={"postcommit_check": "True", "target_platform": "windows"},
            owner="SANDBOX"
        ).first()
        dependant_resource = test_task_2.TestTask2Resource.find(
            attrs={"custom_platform": "windows", "released": "stable"}, owner="SANDBOX", state=ctr.State.READY
        ).first().id
        task_id = test_task_2.TestTask2(
            self,
            description="{}: windows to release with ramdrive".format(description),
            owner=self.Parameters.owner,
            ramdrive=10,
            create_ramdir_on_prepare=True,
            create_ramdir_on_execute=True,
            dependent_resource=dependant_resource,
            dependent_resources=[dependant_resource],
            auto_resource=dependant_resource,
            __requirements__={"tasks_resource": tasks_resource, "client_tags": ctc.Tag.WINDOWS}
        ).enqueue().id
        self.Context.tasks_to_release.append(task_id)
        yield task_id

        task_id = test_task_2.TestTask2(
            self,
            description="{}: python3 to release".format(description),
            __requirements__={"tasks_resource": self.python3_binary},
            use_cgroup=True,
            create_sub_task=True,
            pass_resources_to_subtasks=True,
            sleep_in_subprocess=True,
            create_resource_on_enqueue=True,
            modify_resource=True,
            create_ramdir_on_prepare=True,
            create_ramdir_on_execute=True,
            check_ssh_agent=True,
            sync_resource_in_subprocess=True,
            **self.test_task_2_common_kwargs(test_mounting=True, use_arc=True)
        ).enqueue().id
        self.Context.tasks_to_release.append(task_id)
        yield task_id

    def _create_test_mds_upload_task(self, description):
        yield test_mds_upload.TestMdsUpload(self, description="{}: upload to MDS".format(description)).enqueue().id

    def _create_child_tasks_and_get_ids(self, res_id, res2_id):
        description = "Sandbox acceptance subtask"
        return list(common.utils.chain(
            self._create_serp_test_task(description),
            self._create_deploy_binary_task(description),
            self._create_yabs_tasks(description),
            self._create_autocheck_tasks(description),
            self._create_test_tasks(description, res_id),
            self._create_tasks_for_multislot_client(description),
            self._create_tasks_for_multislot_porto_client(description),
            self._create_privileged_test_tasks_2(description, res2_id),
            self._create_tasklet_acceptance(description),
            self._create_test_tasks_2_without_unmount(description),
            self._create_task_for_nonlxc_client(description),
            self._create_test_task_to_release(description),
            self._create_test_tasks_with_custom_tasks_archive(description),
            self._create_taskboxed_test_tasks(description),
            self._create_acceptance_tasks(description),
            self._create_test_mds_upload_task(description),
        ))

    def _notify_juggler(self, text, status):
        self.server.notification(
            body=text,
            recipients=["host=sandbox.production.regular_acceptance&service=broken"],
            transport=ctn.Transport.JUGGLER,
            check_status=status
        )

    def on_execute(self):
        local_res_id = self.Context.local_resource_id
        local_res2_id = self.Context.local_resource2_id
        if not local_res_id or not local_res2_id:
            res, res2 = self._create_local_resources()
            self._test_resource_download(res)
            self.Context.local_resource_id = local_res_id = res.id
            self.Context.local_resource2_id = local_res2_id = res2.id

        waited_statuses = set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))
        child_tasks_ids = self.Context.child_tasks_ids
        if not child_tasks_ids:
            self.Context.child_tasks_ids = child_tasks_ids = self._create_child_tasks_and_get_ids(
                local_res_id, local_res2_id
            )
            raise sdk2.WaitTask(child_tasks_ids, waited_statuses, wait_all=True)
        elif self.Context.tasks_to_release:
            for task_id in self.Context.tasks_to_release:
                task = self.server.task[task_id][:]
                if task["status"] != ctt.Status.SUCCESS:
                    raise common.errors.TaskFailure("Task #{} is in status {}".format(
                        task_id, task["status"]
                    ))
                self.server.release(
                    task_id=task_id,
                    type=ctt.ReleaseStatus.TESTING,
                    subject="Releasing acceptance task"
                )
            tasks_to_release = self.Context.tasks_to_release
            self.Context.tasks_to_release = None
            raise sdk2.WaitTask(
                tasks_to_release,
                common.utils.chain(ctt.Status.RELEASED, ctt.Status.NOT_RELEASED, ctt.Status.Group.BREAK)
            )
        else:
            subtasks = self.server.task.read(
                id=child_tasks_ids, children=True, hidden=True, limit=len(child_tasks_ids)
            )["items"]
            ok_statuses = list(common.utils.chain(ctt.Status.Group.SUCCEED, ctt.Status.DELETED))
            ok_statuses.remove(ctt.Status.NOT_RELEASED)
            failed_subtasks = [_ for _ in subtasks if _["status"] not in ok_statuses]
            if failed_subtasks:
                self._notify_juggler(
                    "Acceptance task https://sandbox-prestable.yandex-team.ru/task/{}/view failed".format(self.id),
                    ctn.JugglerStatus.CRIT
                )
                lines = ["Failed subtasks:"]
                for subtask in failed_subtasks:
                    subtask_id = subtask["id"]
                    lines.append("#{} of type {}: {}".format(subtask_id,  subtask["type"], subtask["status"]))
                raise common.errors.TaskFailure("\n".join(lines))
            self._notify_juggler("Acceptance {} ok".format(self.id), ctn.JugglerStatus.OK)
