import logging

from sandbox import sdk2

from sandbox.common import rest
from sandbox.common import errors
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt

from sandbox.projects.common import binary_task

logger = logging.getLogger(__name__)


class SandboxAcceptanceRunner(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """Run sandbox acceptance task in prestable installation"""

    PREPROD_TOKEN_NAME = "preprod-oauth-token"
    PREPROD_API_URL = "https://www-sandbox1.n.yandex-team.ru/api/v1.0"

    class Parameters(sdk2.Parameters):
        kill_timeout = 600
        ext_params = binary_task.binary_release_parameters(stable=True)

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        disk_space = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        try:
            preprod_token = sdk2.Vault.data(self.owner, self.PREPROD_TOKEN_NAME)
        except errors.VaultError:
            logging.error("Failed to get preprod OAuth token", exc_info=True)
            return
        client = rest.Client(self.PREPROD_API_URL, preprod_token)
        with self.memoize_stage.run_prestable_sandbox_acceptance:
            subtask = client.task.create(template_alias="sandbox_acceptance", owner="SANDBOX")
            task_id = subtask["id"]
            start_result = client.batch.tasks.start.update([task_id])[0]
            status, message = map(start_result.get, ("status", "message"))
            if status == ctm.BatchResultStatus.ERROR:
                raise errors.TaskNotEnqueued("Task #{} is not enqueued: {}".format(self.id, message))
            self.Context.subtask_id = task_id

        subtask = client.task.read(limit=1, id=self.Context.subtask_id, fields=["status"])
        status = subtask["items"][0]["status"]
        if status in ctt.Status.Group.FINISH + ctt.Status.Group.BREAK:
            if status == ctt.Status.SUCCESS:
                return
            raise errors.TaskFailure("Task #{} in status #{}", self.Context.subtask_id, status)

        raise sdk2.WaitTime(180)

    @sdk2.header(title="Link to prestable task")
    def header(self):
        if self.Context.subtask_id:
            return "<a href=https://sandbox-prestable.yandex-team.ru/task/{}/view>SANDBOX_ACCEPTANCE link</a>".format(self.Context.subtask_id)
        else:
            return "Not started yet"
