import logging

from sandbox import sdk2
import sandbox.common.types.notification as ctn


class SandboxApiQuotaMonitor(sdk2.Task):
    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group("Main settings") as main_settings:
            duration = sdk2.parameters.Integer("Duration for monitoring in minutes", default=60)
            quantile = sdk2.parameters.Integer("API consumption quantile in percents", default=95)
            max_usage = sdk2.parameters.Integer("Maximum usage in percents", default=80)
            notification_mails = sdk2.parameters.List("List of emails to send notifications")

        with sdk2.parameters.Group("Clickhouse settings") as clickhouse_settings:
            clickhouse_db_name = sdk2.parameters.String("Clickhouse database name", required=True)
            clickhouse_table_name = sdk2.parameters.String("Clickhouse table name", required=True)
            clickhouse_db_url = sdk2.parameters.String("Clickhouse database url", required=True)
            clickhouse_username = sdk2.parameters.String("Clickhouse username", required=True)
            clickhouse_password = sdk2.parameters.Vault("Vault record for clickhouse password")

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        from infi.clickhouse_orm import database

        clickhouse_db = database.Database(
            db_name=self.Parameters.clickhouse_db_name,
            db_url=self.Parameters.clickhouse_db_url,
            username=self.Parameters.clickhouse_username,
            password=self.Parameters.clickhouse_password.data(),
            readonly=True,
            autocreate=False
        )

        users = []
        row_request = (
            "SELECT toString(login) as login, quantile({quantile})(consumption) as cons, max(quota) as max_quota "
            "FROM {database}.{table} WHERE timestamp >= now() - {duration} GROUP BY login".format(
                quantile=int(self.Parameters.quantile) / 100.0,
                database=self.Parameters.clickhouse_db_name,
                table=self.Parameters.clickhouse_table_name,
                duration=int(self.Parameters.duration) * 60
            )
        )

        logging.info("Request to clickhouse database: %s", row_request)

        for row in clickhouse_db.select(row_request):
            if row.cons > int(row.max_quota * (int(self.Parameters.max_usage) / 100.0)):
                users.append(row.login)

        info = "Users with API quota consumption more than {} percents of quota: {}".format(
            self.Parameters.max_usage, ", ".join(users)
        )

        logging.info(info)

        if users and self.Parameters.notification_mails:
            self.server.notification(
                subject="API consumption monitoring",
                body=info,
                recipients=self.Parameters.notification_mails,
                transport=ctn.Transport.EMAIL
            )
