import os
import logging

from sandbox import common


class UbuntuRelease(common.utils.Enum):
    common.utils.Enum.preserve_order()
    common.utils.Enum.lower_case()

    LUCID = None
    PRECISE = None
    TRUSTY = None
    XENIAL = None
    BIONIC = None
    FOCAL = None


def file2list(filename):
    if common.system.inside_the_binary():
        from library.python import resource
        path = os.path.join(os.path.dirname(__file__), "config", filename)
        resource_data = resource.find(path)
        if resource_data is None:
            raise IOError("No such resource: {!r}".format(path))
        logging.debug("Arcadia resource %r loaded", path)
        return [line.strip() for line in resource_data.splitlines()]
    else:
        filename = os.path.join(os.path.dirname(os.path.realpath(__file__)), "config", filename)
        with open(filename) as fh:
            return [line.strip() for line in fh]


class Bootstrap(object):

    REQUIREMENTS = ["debootstrap"]
    COMPONENTS = ["main", "restricted", "universe", "multiverse"]

    def __init__(self, ubuntu_release, basic_includes_only):
        self.ubuntu_release = ubuntu_release
        self.basic_includes_only = basic_includes_only

    def cmd(self, rootfs):
        return "debootstrap --components={} --arch=amd64 --exclude={} --include={} {} {} {}".format(
            ",".join(self.COMPONENTS),
            ",".join(self._excludes),
            ",".join(self._includes),
            self.ubuntu_release,
            rootfs,
            self._mirror,
        )

    @property
    def _includes(self):
        includes = file2list("deb_packages_basic")
        if self.ubuntu_release == UbuntuRelease.FOCAL:
            includes.append("perl-openssl-defaults")
        if self.basic_includes_only:
            return includes
        if self.ubuntu_release not in (UbuntuRelease.LUCID, UbuntuRelease.BIONIC, UbuntuRelease.FOCAL):
            includes.extend(file2list("deb_packages_common"))
        return includes + file2list("deb_packages_specific.{}".format(self.ubuntu_release))

    @property
    def _excludes(self):
        return file2list("deb_packages_exclude")

    @property
    def _mirror(self):
        if self.ubuntu_release == UbuntuRelease.LUCID:
            return "http://dist.yandex.ru/ubuntu/"
        if self.ubuntu_release == UbuntuRelease.PRECISE:
            return "http://mirror.yandex.ru/old-ubuntu/"
        return "http://mirror.yandex.ru/ubuntu/"


def file_descr(uid=0, gid=0, mask=0o0644, content=None):
    return {"owner": uid, "group": gid, "mask": mask, "content": content or [""]}


class Image(object):

    BASE_PACKAGES = ["yandex-ubuntu-archive-apt", "yandex-archive-keyring", "yandex-search-common-apt", "rsync"]

    def __init__(self, ubuntu_release, base_packages, custom_packages):
        self.ubuntu_release = ubuntu_release
        self._base_packages = base_packages
        self._custom_packages = custom_packages

    @property
    def base_sources(self):
        if self.ubuntu_release == UbuntuRelease.LUCID:
            main_source = security_source = updates_source = "http://dist.yandex.ru/ubuntu"
        elif self.ubuntu_release == UbuntuRelease.PRECISE:
            main_source = security_source = updates_source = "http://mirror.yandex.ru/old-ubuntu/"
        else:
            main_source = "http://mirror.yandex.ru/ubuntu"
            security_source = updates_source = "http://security.ubuntu.com/ubuntu"

        return {
            "/etc/apt/sources.list": [
                "deb {} {} main restricted universe multiverse".format(main_source, self.ubuntu_release)
            ],
            "/etc/apt/sources.list.d/ubuntu-security.list": [
                "deb {} {}-security main universe multiverse".format(security_source, self.ubuntu_release)
            ],
            "/etc/apt/sources.list.d/ubuntu-updates.list": [
                "deb {} {}-updates main restricted universe multiverse".format(updates_source, self.ubuntu_release)
            ]
        }

    @property
    def base_packages(self):
        if self._base_packages:
            return self._base_packages
        return self.BASE_PACKAGES

    @property
    def extra_packages(self):
        suffix = None
        if self._custom_packages:
            return self._custom_packages
        if self.ubuntu_release in (UbuntuRelease.LUCID, UbuntuRelease.FOCAL):
            suffix = ".{}".format(self.ubuntu_release)
        return file2list(
            "deb_packages_extra" + (suffix if suffix else "")
        )

    @property
    def specific_conf_files(self):
        if self.ubuntu_release == UbuntuRelease.LUCID:
            return {
                "/etc/apt/sources.list.d/yandex-common-all.list":
                    file_descr(content=file2list("yandex-common-all.lucid"))
            }
        return {}

    @property
    def conf_files(self):
        files = {
            "/etc/hostname": file_descr(),
            "/etc/resolv.conf": file_descr(mask=0o0664),
            "/root/.ssh/authorized_keys": file_descr(mask=0o0600),
            "/etc/network/interfaces": file_descr(content=file2list("interfaces")),
            "/etc/rc.local": file_descr(mask=0o0755, content=file2list("routing.sh")),
            "/etc/ssh/sshd_config": file_descr(content=file2list("sshd_config")),
            "/etc/ssh/ssh_config": file_descr(content=file2list("ssh_config")),
            "/etc/security/capability.conf": file_descr(content=file2list("capability.conf")),
        }
        files.update(self.specific_conf_files)
        return files


def test_image_statically(rootfs):
    home_dir = os.path.join(rootfs, "home")
    assert os.path.isdir(home_dir) and not os.path.islink(home_dir), "/home must be a directory, not the link"
