import re
import shlex
import logging
import datetime as dt

from sandbox import sdk2
from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.sandbox import sandbox_lxc_image


class SandboxLxcImagePackageUpdater(sdk2.Task):
    """
    Check for a new version of the packages.
    If a new version of any package is available - rebuild LXC image.
    If ubuntu_release parameter is 'all' - run itself for all available release versions
    as a child task and wait for the result.
    """

    PACKAGES = [
        "tzdata"
    ]

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.LXC
        privileged = True
        disk_space = 1024

    class Parameters(sdk2.Task.Parameters):
        packages = sdk2.parameters.List("Additional packages to check for update", default=[])

        with sdk2.parameters.String("Ubuntu release", default="all") as ubuntu_release:
            ubuntu_release.values["all"] = "All available releases"
            # noinspection PyTypeChecker
            for release in sandbox_lxc_image.UbuntuRelease:
                ubuntu_release.values[release] = release

        with sdk2.parameters.Group("Common Parameters") as common_block:
            notify_via_telegram = sdk2.parameters.Bool("Notify via Telegram when a new package version is available")

            with notify_via_telegram.value[True]:
                telegram_chat_ids = sdk2.parameters.List("Telegram chat identifiers")
                telegram_bot_token = sdk2.parameters.Vault(
                    "Vault record <owner>:<name> with a Telegram bot token",
                    default="SANDBOX:yasandboxbot-telegram-token"
                )

    def on_enqueue(self):
        if self.Parameters.ubuntu_release != "all":
            self.Requirements.client_tags &= ctc.Tag["LINUX_{}".format(self.Parameters.ubuntu_release.upper())]

    def on_execute(self):
        if self.Parameters.ubuntu_release == "all":
            self.update_all_images()
        else:
            self.update_current_image()

    def update_all_images(self):
        with self.memoize_stage.run_child_tasks:
            self.Context.child_task_ids = []

            for release in sandbox_lxc_image.UbuntuRelease:
                if release == sandbox_lxc_image.UbuntuRelease.LUCID:
                    continue
                task = SandboxLxcImagePackageUpdater(
                    self,
                    description="Check for a new version of the packages for {}".format(release),
                    ubuntu_release=release,
                    packages=self.Parameters.packages,
                    notify_via_telegram=self.Parameters.notify_via_telegram,
                    telegram_chat_ids=self.Parameters.telegram_chat_ids,
                    telegram_bot_token=str(self.Parameters.telegram_bot_token),
                )
                task.enqueue()
                self.Context.child_task_ids.append(task.id)

            raise sdk2.WaitTask(self.Context.child_task_ids, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

        with self.memoize_stage.check_children_state:
            for child_task in self.find(id=self.Context.child_task_ids):
                if child_task.status != ctt.Status.SUCCESS:
                    raise common.errors.TaskFailure(
                        "{} image update failed".format(child_task.Parameters.ubuntu_release)
                    )

    def update_current_image(self):
        with self.memoize_stage.build_lxc_image(commit_on_entrance=False, commit_on_wait=True):
            new_packages = self._check_for_new_package_version(set(self.Parameters.packages + self.PACKAGES))
            if new_packages:
                self._notify(new_packages)

                task = sandbox_lxc_image.SandboxLxcImage(
                    self,
                    owner=self.owner,
                    description="update package versions: {}".format(self._get_packages_summary_str(new_packages)),
                    ubuntu_release=self.Parameters.ubuntu_release,
                    test_result_lxc=True,
                    custom_image=False,
                    push_to_preprod=True,
                )
                self.Context.lxc_image_build_task_id = task.id

                task.enqueue()
                raise sdk2.WaitTask(task.id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)
            else:
                logging.info("All the packages are up-to-date")
                return

        with self.memoize_stage.release_task:
            lxc_image_build_task = self.find(id=self.Context.lxc_image_build_task_id).first()
            if lxc_image_build_task.status != ctt.Status.SUCCESS:
                raise common.errors.TaskFailure("Lxc image build task {} failed".format(lxc_image_build_task.id))

    def _execute(self, command, **kwargs):
        logger = logging.getLogger("execute")

        with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
            stdout = sp.check_output(shlex.split(command), stderr=pl.stderr, **kwargs)

        logger.debug(stdout)
        return stdout

    @classmethod
    def _get_packages_summary_str(cls, packages, sep=", "):
        return sep.join([
            "{} (current: {}, candidate: {})".format(name, current, candidate)
            for name, (current, candidate) in packages.items()
        ])

    @classmethod
    def _re_search(cls, pattern, string):
        match = re.search(pattern, string)
        if match is None:
            raise common.errors.TaskError("Unexpected command output: '{}' pattern not found".format(pattern))
        else:
            return match.group(1)

    def _check_for_new_package_version(self, packages):
        result = {}

        self._execute("apt-get update -y")

        for package in packages:
            stdout = self._execute("apt-cache policy {}".format(package))

            installed = self._re_search(r"Installed: (.*)\n", stdout)
            candidate = self._re_search(r"Candidate: (.*)\n", stdout)

            if candidate != "(none)" and installed != candidate:
                result[package] = (installed, candidate)

        return result

    def _notify(self, packages):
        message = "New package versions are available for {} release:\n{}".format(
            self.Parameters.ubuntu_release,
            self._get_packages_summary_str(packages, sep="\n")
        )
        logging.info(message)

        if self.Parameters.notify_via_telegram:
            token = self.Parameters.telegram_bot_token.data()
            for chat_id in self.Parameters.telegram_chat_ids:
                self._notify_telegram_chat(chat_id, token, message)

    def _notify_telegram_chat(self, chat_id, token, message):
        message = '<strong>{}</strong> {}\nTask: <a href="{}">{}</a>'.format(
            str(dt.date.today()), message, common.utils.get_task_link(self.id), self.id
        )

        try:
            bot = common.telegram.TelegramBot(token)
            bot.send_message(chat_id, message, common.telegram.ParseMode.HTML)
        except Exception:
            err_message = "Failed to send a chat notification"
            logging.exception(err_message)
            raise common.errors.TemporaryError(err_message)
