# coding: utf-8

import time
import logging
import subprocess

from pymongo import MongoClient, errors as mongoerr
from functools import wraps

from sandbox import common
import sandbox.common.types.task as ctt

from sandbox.sandboxsdk.task import SandboxTask


class SandboxMongoDefrag(SandboxTask):
    """
    Manager
    0. Resolves conductor tag
    1. Sotrs hosts and move PRIMARY to the end
    2. Start child task
    Worker
    0. Check if instance is a primary, than sends "stepDown"
    1. Stops all mongodb instances.
    2. Remove all from dbpath
    3. Start all mongodb instances
    """

    type = "SANDBOX_MONGO_DEFRAG"
    execution_space = 50

    @common.utils.singleton_classproperty
    def fqdn(self):
        return common.config.Registry().this.fqdn

    def retry(ExceptionToCheck, tries=4, delay=3, backoff=2):
        """Retry calling the decorated function using an exponential backoff.
        Original: https://github.com/saltycrane/retry-decorator/blob/master/decorators.py
        """

        def deco_retry(f):
            @wraps(f)
            def f_retry(*args, **kwargs):
                mtries, mdelay = tries, delay
                while mtries > 1:
                    try:
                        return f(*args, **kwargs)
                    except ExceptionToCheck:
                        time.sleep(mdelay)
                        logging.info('Retrying...')
                        mtries -= 1
                        mdelay *= backoff
                return f(*args, **kwargs)

            return f_retry  # true decorator

        return deco_retry

    def check_primary(self, port):
        """
        :param port: mongo replica port
        :return: True for master, False in other cases
        """
        logging.info('Checking primary on {0} : {1}'.format(self.fqdn, port))
        try:
            with MongoClient("localhost", port) as client:
                result = client.is_primary
                if result:
                    logging.info('{0} : {1} is primary'.format(self.fqdn, port))
                else:
                    logging.info('{0} : {1} is secondary'.format(self.fqdn, port))
                return result
        except mongoerr.ConnectionFailure:
            logging.info('{0} : {1} is secondary'.format(self.fqdn, port))
            return False

    @retry(mongoerr.ConnectionFailure)
    def check_rs_status(self, port):
        with MongoClient("localhost", port) as client:
            rs_hosts = client.admin.command({"replSetGetStatus": "1"})["members"]
        for host in rs_hosts:
            if host["name"] == "{0}:{1}".format(self.fqdn, port):
                return host["stateStr"]

    @retry(mongoerr.ConnectionFailure)
    def stepdown(self, port):
        try:
            with MongoClient("localhost", port) as client:
                client.admin.command({"replSetStepDown": "30"})
        except Exception:
            pass
        logging.info('Have sent rs.stepDown() on {0} : {1}'.format(self.fqdn, port))
        if not self.check_primary(port):
            logging.info('rs.stepDown() done on {0} : {1}'.format(self.fqdn, port))
            return True
        else:
            logging.error('StepDown failed on {0} : {1}'.format(self.fqdn, port))
            return False

    @retry(common.errors.TaskFailure)
    def service_mongodb(self, port, action):
        logging.info("EXEC: {1} mongodb on {0} port".format(port, action))
        process = subprocess.Popen(
            ["sudo", "/etc/init.d/mongodb_{0}".format(port), action],
            stderr=subprocess.STDOUT,
            stdout=subprocess.PIPE,
        )
        out = process.communicate()[0]
        if process.returncode != 0:
            raise common.errors.TaskFailure("cleanup_dbpath failed")
        logging.info(out)

    def cleanup_dbpath(self, port):
        logging.info("EXEC: cleanup dbpath for {0}:{1}".format(self.fqdn, port))
        logging.info("EXEC: sudo /bin/rm -rf /ssd/mongodbs/db_{0}/* -v".format(port))
        process = subprocess.Popen(
            ["sudo", "/bin/rm", "-rf", "/ssd/mongodbs/db_{0}/*".format(port), "-v"],
            stderr=subprocess.STDOUT,
            stdout=subprocess.PIPE,
        )
        out = process.communicate()[0]
        if process.returncode != 0:
            raise common.errors.TaskFailure("cleanup_dbpath failed")
        logging.info(out)

    def worker_exec(self):
        for mongo_port in range(37001, 37011):
            if self.check_primary(mongo_port):
                while self.stepdown(mongo_port) is not True:
                    time.sleep(3)

        for mongo_port in range(37001, 37011):
            self.service_mongodb(mongo_port, "stop")
        for mongo_port in range(37001, 37011):
            self.cleanup_dbpath(mongo_port)
        for mongo_port in range(37001, 37011):
            self.service_mongodb(mongo_port, "start")

        time.sleep(15)
        shards_ok = False
        while shards_ok is False:
            for mongo_port in range(37001, 37011):
                if self.check_rs_status(mongo_port) == "SECONDARY":
                    shards_ok = True
                else:
                    shards_ok = False
            time.sleep(60)

    def manager_exec(self):
        # data = common.rest.Client().service.status.database.shards.read()
        shards = self.ctx.get("shards")
        if shards is None:
            shards = self.ctx["shards"] = self.getallshards()
        while shards:
            srv = shards.pop(0)
            task = self.create_subtask("SANDBOX_MONGO_DEFRAG", host=srv, description="Run mongo defrag")
            self.wait_tasks(task, ctt.Status.SUCCESS)

    def on_execute(self):
        self.worker_exec()
