import os
import json
import logging
import subprocess

import six

from sandbox import sdk2

from sandbox.projects.common.arcadia import sdk as arcadia_sdk


class SandboxSortYtTables(sdk2.Task):
    """
    Sort Yt tables by given paths
    """

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 2048
        disk_space = 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # no shared caches

    class Parameters(sdk2.Parameters):
        description = "Sort YT tables by given paths"

        paths = sdk2.parameters.Dict("Dict of mapping path in YT to key for sorting")
        proxy = sdk2.parameters.String("Yt cluster", default="hahn.yt.yandex.net", required=True)
        yt_secret = sdk2.parameters.YavSecretWithKey("Secret in YAV with access to YT.",required=True)

    def call_yt_command(self, cmd, env=None, as_json=True):
        if as_json:
            cmd += ["--format", "json"]
        logging.info("Run command: %s", cmd)
        p = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, env=env)
        result, err = p.communicate()
        if p.returncode:
            logging.error("Command {} failed. Stdout: {}. Stderr: {}".format(cmd, result, err))
            return None
        if as_json:
            result = json.loads(result.strip())
        logging.info("Command result: %s", result)
        return result

    def process_directory(self, ya_path, dir_name, sort_key, yt_token):
        env = os.environ.copy()
        env["YT_TOKEN"] = yt_token
        tables = self.call_yt_command([
            ya_path, "tool", "yt", "get", "--path", dir_name, "--proxy",
            self.Parameters.proxy, "--attribute", "sorted"
        ], env=env, as_json=True)
        if tables is None:
            logging.info("Can't get information about directory %s", dir_name)
            self.set_info("Can't get information about directory {}".format(dir_name))
            return
        unsorted_tables = [
            table_name for table_name, table in six.iteritems(tables) if not table["$attributes"]["sorted"]
        ]
        unsorted_tables.sort()
        for table_name in unsorted_tables[:-2]:
            full_table_name = "/".join((dir_name, table_name))
            result = self.call_yt_command([
                ya_path, "tool", "yt", "sort", "--src", full_table_name, "--dst", full_table_name,
                "--sort-by", sort_key, "--proxy", self.Parameters.proxy
            ], env=env, as_json=False)
            if result is None:
                logging.info("Failed to sort table %s", full_table_name)
                self.set_info("Failed to sort table {}".format(full_table_name))

    def on_execute(self):
        yt_token = self.Parameters.yt_secret.value(self.Parameters.yt_secret.default_key)
        with arcadia_sdk.mount_arc_path("arcadia-arc:/#trunk") as arcadia_dir:
            ya_path = os.path.join(arcadia_dir, "ya")
            for dir_name, sort_key in six.iteritems(self.Parameters.paths):
                self.process_directory(ya_path, dir_name, sort_key, yt_token)
