import logging

from sandbox import common
from sandbox import sdk2
import sandbox.common.types.task as ctt
import sandbox.projects.sandbox.cleanup_2 as cleanup


class ServiceSandboxClients2(sdk2.Task):
    """
    Schedules 'CLEANUP_2' task on each Sandbox host with the lowest priority.
    """

    class Parameters(sdk2.Task.Parameters):
        tags_query = sdk2.parameters.String("Client tags query to limit scope for cleanup activity", default_value="")

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Context):
        children_tasks_ids = []

    def on_execute(self):
        if self.Context.children_tasks_ids:
            return

        sandbox = common.rest.Client()
        clients = []
        pos = 0
        chunk = 100
        count = sandbox.client.read(alive=True, limit=0, tags=self.Parameters.tags_query).get("total", 0)
        while pos <= count:
            response = sandbox.client.read(
                alive=True, offset=pos, limit=chunk, fields=["id", "platform"], tags=self.Parameters.tags_query
            )
            clients.extend(response.get("items", []))
            pos += chunk
            count = response.get("total", count)

        children_task_ids = []
        for client in clients:
            host = client["id"]
            logging.info("Create tasks for host %s" % host)

            cleanup_task = cleanup.Cleanup2(
                self,
                owner=self.owner,
                description="Cleanup host {}".format(host),
                priority=ctt.Priority(ctt.Priority.Class.BACKGROUND, ctt.Priority.Subclass.LOW),
                kill_timeout=10 * 60 * 60,
                run_in_dry_mode=False,
                semaphore="sandbox/cleanup/regular"
            )
            cleanup_task.Requirements.host = host
            cleanup_task.Requirements.disk_space = 3 * 1024
            result = cleanup_task.save()
            logging.debug("Task create result %s", str(result))
            task_id = cleanup_task.enqueue().id
            children_task_ids.append(task_id)

        result = self.server.batch.tasks.start.update(children_task_ids)
        self.Context.children_tasks_ids.append(children_task_ids)
        logging.debug('Subtasks start result {}'.format(result))
