import json
import os.path
import itertools
import collections
import operator as op

from sandbox import common

from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.paths import make_folder

from sandbox import projects
from sandbox.projects import resource_types


class LastRunDaysAgo(parameters.SandboxIntegerParameter):
    name = "last_run_days_ago"
    description = "The number of days since the last run tasks of each type"
    default_value = 180
    required = True


class SendTo(parameters.SandboxStringParameter):
    name = "send_to"
    description = "Recepients nicknames separated with comma"
    default_value = ""


class ResultDir(object):
    def __init__(self, name):
        self.path = make_folder(name)

    def write_json_file(self, fname, data):
        fname = fname if fname.endswith(".json") else u"{}.json".format(fname)
        with open(os.path.join(self.path, fname), "w") as f:
            json.dump(data, f, indent=2, sort_keys=True)

    def write_text_file(self, fname, data):
        if isinstance(data, unicode):
            data = data.encode("utf-8")
        with open(os.path.join(self.path, fname), "w") as f:
            f.write(data)


EXCLUSIONS = [
    'BUILD_BASESEARCH_STUB',
    'BUILD_CMAKE',
    'BUILD_CMAKE_FOR_ALL',
    'BUILD_FIX_QUERIES',
    'BUILD_GPERFTOOLS',
    'BUILD_IDX_URL_DUPS',
    'BUILD_MN_CUDA',
    'BUILD_NEWS_ARCHIVE_SHARD',
    'BUILD_NEWS_MEMCACHED',
    'BUILD_NEWS_NGINX',
    'BUILD_NEWS_REDIS_SERVER',
    'BUILD_NINJA',
    'BUILD_NINJA_FOR_ALL',
    'BUILD_PUMPKIN_RESINFOD_BUNDLE',
    'BUILD_QURL_RATES',
    'BUILD_REPORT',
    'BUILD_RTYSERVER_DEB',
    'BUILD_SVN',
    'BUILD_SVN_FOR_ALL',
    'CHECK_RUNNING_VIRTUAL_MACHINES',
    'COMPARE_PROFILE_STATS',
    'DUMP_EVENTLOG',
    'IMAGES_TEST_BASESEARCH_CGI_MEMCHECK',
    'IMAGES_TEST_BASESEARCH_CGI_PARAMS',
    'IMAGES_TEST_MIDDLESEARCH_MASSIF',
    'RELEASE_CYCOUNTER_FILES',
    'RELEASE_LOOP',
    'REPORT_RULE_TEST_FULL',
    'TEST_OPENSTACK',
    'TEST_REALSEARCH_PERFORMANCE_BEST',
    'TEST_RTYSERVER',
    'TEST_YAMR',
]


def prepare_row_data(row_data):
    for cell in row_data:
        if isinstance(cell, basestring):
            yield cell
        elif isinstance(cell, collections.Iterable):
            yield ", ".join(cell)
        else:
            yield cell


def render_row(row_data, td='td'):
    row_tpl = "<tr><{0}>{{}}</{0}></tr>".format(td)
    td_tpl = "</{0}><{0}>".format(td)

    return row_tpl.format(td_tpl.join(prepare_row_data(row_data)))


def render_header(row_data):
    return render_row(row_data, td='th')


def render_body(data, predicate):
    if data:
        return '\n'.join(render_row(predicate(line)) for line in data)
    else:
        return u"Not present"


def render_table(headers, data, predicate):
    return "<table><thead>{}</thead><tbody>{}</tbody></table>".format(
        render_header(headers),
        render_body(data, predicate)
    )


def email_render_table(title, headers, predicate, data):
    return "<h1>{} (count: {})</h1> {}".format(
        title,
        len(data),
        render_table(headers, data, predicate)
    )


def build_email(last_run_days_ago, content, url):

    return u"""<!DOCTYPE html>
<html>
    <body>
        <p>The types of tasks that do not run more than <b>{days_ago}</b> days.</p>
        <a href="{url}">Link to Sandbox resource</a>
        {content}
    </body>
</html>
""".format(
        days_ago=last_run_days_ago,
        url=url,
        content=content
    )


TaskType = collections.namedtuple('TaskType', ['name', 'last_run', 'is_code_exists', 'owners', 'childs'])


class ServiceSandboxOutdatedTaskTypes(task.SandboxTask):
    """
    Find all task types run more than six months ago.
    """

    SERVICE = True
    type = "SERVICE_SANDBOX_OUTDATED_TASKS"

    input_parameters = (
        LastRunDaysAgo,
        SendTo
    )

    def on_execute(self):
        out_dir = ResultDir("result")
        days_ago = self.ctx[LastRunDaysAgo.name]

        out_dir.write_json_file("exclusions", sorted(EXCLUSIONS))

        parent_task_types = collections.defaultdict(list)
        for parent_tt_name, parent in projects.TYPES.iteritems():
            for child_tt_name, child in projects.TYPES.iteritems():
                if (child.cls is not parent.cls) and issubclass(child.cls, parent.cls):
                    parent_task_types[parent_tt_name].append(child_tt_name)

        for v in parent_task_types.itervalues():
            v.sort()

        out_dir.write_json_file("parent_task_types", parent_task_types)

        rest_client = common.rest.Client()
        abandoned_task_types = rest_client.service.statistics.task.types.not_used.read(days_ago=days_ago)
        trigged_task_types = rest_client.service.statistics.task.types.not_used.read(days_ago=-1)

        exc_set = set(EXCLUSIONS)

        def bound_api_result(task_types):
            for tt in task_types:
                task_type_name = tt["type"]
                last_run = tt["last_run"]

                if task_type_name in exc_set:
                    continue

                task_type_location = projects.TYPES.get(task_type_name)
                is_code_exists = task_type_location is not None

                yield TaskType(
                    task_type_name, last_run, is_code_exists,
                    sorted(task_type_location.cls.owners) if is_code_exists and task_type_location.cls.owners else [],
                    parent_task_types.get(task_type_name, [])
                )

        abandoned = bound_api_result(abandoned_task_types)
        abandoned = sorted(abandoned, key=op.attrgetter('is_code_exists'))

        abandoned_by_code = dict(
            (key, sorted(grp, key=op.attrgetter('name'))) for key, grp in itertools.groupby(
                abandoned, key=op.attrgetter('is_code_exists')
            )
        )

        abandoned_task_types_w_code = abandoned_by_code.get(True, [])
        abandoned_task_types_wo_code = abandoned_by_code.get(False, [])

        out_dir.write_json_file("outdated_task_types_with_code", abandoned_task_types_w_code)
        out_dir.write_json_file("outdated_task_types_without_code", abandoned_task_types_wo_code)

        never_runs_names = set(projects.TYPES) - set(tt["type"] for tt in trigged_task_types)
        never_runs = bound_api_result({"type": name, "last_run": None} for name in never_runs_names)
        never_runs = sorted(never_runs, key=op.attrgetter('name'))

        out_dir.write_json_file("never_runs_task_types", never_runs)

        resource = self._create_resource(
            "Outdated task types report",
            out_dir.path,
            resource_types.SERVICE_SANDBOX_OUTDATED_TASK_TYPES_REPORTS
        )
        resource.mark_ready()

        mail_to = [x.strip() for x in self.ctx[SendTo.name].split(",")]
        if mail_to:
            mail_cc = []
            mail_subject = "Report - Outdated task types"
            content = "".join((
                email_render_table("Not used task types with code",
                                   ["Task type", "Last run", "Owners", "Children"],
                                   lambda item: (item.name, item.last_run, item.owners, item.childs),
                                   abandoned_task_types_w_code),
                email_render_table("Not used task types without code",
                                   ["Task type", "Last run"],
                                   lambda item: (item.name, item.last_run),
                                   abandoned_task_types_wo_code),
                email_render_table("Never runs task types",
                                   ["Task type", "Owners", "Children"],
                                   lambda item: (item.name, item.owners, item.childs),
                                   never_runs)
            ))

            mail_body = build_email(days_ago, content, resource.http_url())
            channel.sandbox.send_email(mail_to, mail_cc, mail_subject, mail_body, content_type="text/html")
