# coding: utf-8

from sandbox import common

from sandbox import projects
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel


class NumberOfSubtasksToRun(parameters.SandboxIntegerParameter):
    name = 'number_of_subtasks_to_run'
    description = 'Number of subtasks to run'
    default_value = 1


class TaskType(parameters.SandboxStringParameter):
    name = 'task_type'
    description = 'Task type'
    required = True
    choices = None


class TaskToCopy(parameters.SandboxIntegerParameter):
    name = 'task_to_copy'
    default_value = 0
    description = 'Task id for copy from (fill with task id and click `Save` button)'


class SubTasksRunner(task.SandboxTask):
    type = 'SUB_TASKS_RUNNER'
    input_parameters = [
        TaskType,
        TaskToCopy,
        NumberOfSubtasksToRun,
    ]

    def __get_subtask_defaults(self, task_type):
        defaults = {}
        try:
            cls = projects.TYPES[task_type].cls
        except KeyError:
            return defaults

        if isinstance(cls.input_parameters, property):
            input_parameters = cls.input_parameters.fget(cls)
        else:
            input_parameters = cls.input_parameters
        for param in input_parameters or ():
            if not isinstance(param.default_value, property):
                defaults[param.name] = param.default_value
        return defaults

    def __update_ctx_from_subtask(self):
        task_id = self.ctx.get(TaskToCopy.name)
        if task_id and task_id != self.ctx.get('copy_of', 0):
            task_to_copy = channel.sandbox.get_task(task_id)
            task_type = task_to_copy.type
            task_to_copy.prio = task_to_copy.priority
            task_to_copy.descr = task_to_copy.description
            task.SandboxTask.init_as_copy(self, task_to_copy)
        else:
            task_type = self.ctx.get(TaskType.name)
        if task_type:
            for k, v in self.__get_subtask_defaults(task_type).iteritems():
                self.ctx.setdefault(k, v)
            self.ctx[TaskType.name] = task_type
        return task_type

    def init_as_copy(self, task_, request=None):
        task.SandboxTask.init_as_copy(self, task_)
        sub_task_type = self.ctx.get(TaskType.name)
        if sub_task_type is not None:
            for name in self.__get_subtask_defaults(sub_task_type):
                if name in task_.ctx:
                    self.ctx[name] = task_.ctx[name]

    def init_context(self):
        self.__update_ctx_from_subtask()
        return task.SandboxTask.init_context(self)

    def on_enqueue(self):
        if self.type == self.ctx[TaskType.name]:
            raise common.errors.TaskError("Cannot be running under {}".format(self.type))

    def on_execute(self):
        sub_tasks = [t for t in channel.sandbox.list_tasks(parent_id=self.id, task_type=self.ctx[TaskType.name]) if not t.is_not_ready()]
        if sub_tasks:
            return
        sub_tasks = []
        for i in xrange(int(self.ctx[NumberOfSubtasksToRun.name])):
            descr = 'Subtask %s of task #%s' % (i, self.id)
            sub_task = self.create_subtask(self.ctx[TaskType.name], descr)
            sub_tasks.append(sub_task)
        self.wait_all_tasks_completed(sub_tasks)
