import collections
import json
import logging
import os
import subprocess
import textwrap
import time
import yaml

import six

from sandbox import sdk2

from sandbox.common import errors as common_errors
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr

from sandbox.projects.sandbox.tasklets import resources as tasklet_resources
from sandbox.projects.common.arcadia import sdk as arcadia_sdk


class Tasklet(object):
    def __init__(self, namespace, name, label):
        self.namespace = namespace
        self.name = name
        self.label = label

    @classmethod
    def from_meta(cls, tasklet):
        tasklet_meta = tasklet.get("meta", {})
        tasklet_spec = tasklet.get("spec", {})
        return cls(
            tasklet_meta.get("namespace"), tasklet_meta.get("name"), tasklet_spec.get("tracking_label")
        )

    @classmethod
    def from_yaml(cls, tasklet):
        tasklet_meta = tasklet.get("meta", {})
        return cls(
            tasklet_meta.get("namespace"), tasklet_meta.get("name"), tasklet_meta.get("tracking_label")
        )

    def validate(self, namespace, namespace_tasklets):
        if not self.namespace or not self.name or not self.label:
            logging.info(
                "Skip tasklet '%s'. Tasklet with namespace: '%s', label: %s has empty field",
                self.name, self.namespace, self.label
            )
            return False
        if self.namespace != namespace:
            logging.info(
                "Skip tasklet '%s': it namespace '%s' != '%s'", self.name, self.namespace, namespace
            )
            return False
        if self.name not in namespace_tasklets:
            logging.info(
                "Skip tasklet '%s': not found in server tasklets: %s'", namespace_tasklets.keys()
            )
        return True

    def full_name(self):
        return "{}/{}:{}".format(self.namespace, self.name, self.label)


ExecutionInput = collections.namedtuple("ExecutionInput", "tasklet input_file")


class Execution(object):
    def __init__(self, execution, tasklet):
        self.execution = execution
        self.tasklet = tasklet

    def finished(self):
        return self.execution["execution"]["status"]["status"] == "E_EXECUTION_STATUS_FINISHED"

    def success(self):
        return self.finished() and not self.execution["execution"]["status"].get("error", None)

    def id(self):
        return self.execution["execution"]["meta"]["id"]

    def link(self):
        return "<a href='https://sandbox.yandex-team.ru/tasks?type=TASKLET&input_parameters={}'>{}</a>".format(
            six.moves.urllib_parse.quote(json.dumps({"execution_id": self.id()})),
            self.id(),
        )


class TaskletCli(object):
    def __init__(self, tool_command, installation, token):
        self.tool_command = tool_command
        self.installation = installation
        self.env = os.environ.copy()
        self.env["TASKLET_TOKEN"] = token
        self.env["TASKLET_DEBUG"] = "1"

    def command(self, cmd):
        tasklet_cmd = self.tool_command + cmd + ["-c", self.installation, "-o", "json"]

        logging.info("Run tasklet tool command: %s", tasklet_cmd)
        p = subprocess.Popen(tasklet_cmd, env=self.env, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        out, err = p.communicate()
        if p.returncode:
            raise common_errors.TaskError("Command {} failed. Stdout: {}. Stderr: {}".format(tasklet_cmd, out, err))
        result = json.loads(out.strip())
        logging.info("Tasklet tool command result: %s", result)
        return result


class Tasklet2Acceptance(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        disk_space = 2

        class Caches(sdk2.Requirements.Caches):
            pass  # no shared caches

    class Parameters(sdk2.Parameters):
        description = textwrap.dedent("""
            Launch acceptance tasklet. Inner join tasklet from 'namespaces' and 'tasklet_arcadia_paths'.
            For every tasklet search all files starts with 'test_input' and run every tasklet with every input file.
        """)
        kill_timeout = 30 * 60  # 30 minutes

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia URL; must always point to Arcadia root (branches/tags are allowed). If empty, trunk URL is used."
        )
        tasklet_cli = sdk2.parameters.Resource(
            "Tasklet tool binary resource. If empty, 'ya tool tasklet' is used"
        )
        installation = sdk2.parameters.String(
            "Tasklet installation", choices=tuple((x, x) for x in ("local", "env", "test", "prod")), default="test"
        )
        namespaces = sdk2.parameters.List(
            "tasklet testing namespace", sdk2.parameters.String, default=["test-tasklets"]
        )
        tasklet_arcadia_paths = sdk2.parameters.List(
            "Paths in arcadia to directory with tasklets",
            sdk2.parameters.String,
            default=["tasklet/experimental/examples"]
        )

    def on_create(self):
        if not self.Requirements.tasks_resource:
            self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
                state=ctr.State.READY,
                owner="TASKLETS",
                attrs={"task_type": self.type.name, "released": ctt.ReleaseStatus.STABLE},
            ).first()

    def get_tasklets_from_namespace(self, tasklet_cli, checked_namespace):
        cache = getattr(self, "tasklets_cache", None)
        if cache is None:
            self.tasklets_cache = {}
            cache = self.tasklets_cache

        if checked_namespace in cache:
            return cache[checked_namespace]

        server_tasklet_list = tasklet_cli.command(["tasklet", "list", checked_namespace]).get("tasklets", [])
        logging.info("Get tasklets from server: %s", server_tasklet_list)
        server_tasklet_dict = {}
        for tasklet_info in server_tasklet_list:
            tasklet = Tasklet.from_meta(tasklet_info)
            server_tasklet_dict[tasklet.name] = tasklet

        cache[checked_namespace] = server_tasklet_dict
        return server_tasklet_dict

    def get_examples_tasklets(self, tasklet_cli, arcadia_dir, checked_namespace, checked_arc_path):
        server_tasklet_dict = self.get_tasklets_from_namespace(tasklet_cli, checked_namespace)
        tasklets = []
        examples_dir = os.path.join(arcadia_dir, checked_arc_path)

        for fname in os.listdir(examples_dir):
            tasklet_path = os.path.join(examples_dir, fname)
            if not os.path.isdir(tasklet_path):
                continue

            tyaml_path = os.path.join(tasklet_path, "t.yaml")
            if not os.path.exists(tyaml_path):
                logging.info("Skip '%s' because t.yaml not found.", tasklet_path)
                continue
            for test_filename in os.listdir(tasklet_path):
                if not test_filename.startswith("test_input"):
                    continue
                input_file = os.path.join(tasklet_path, test_filename)
                if not input_file:
                    logging.info("Skip '%s' because '%s' not found.", tasklet_path, test_filename)
                    continue

                with open(tyaml_path, "rb") as f:
                    tasklet = Tasklet.from_yaml(yaml.load(f.read()))

                if not tasklet.validate(checked_namespace, server_tasklet_dict):
                    continue
                tasklets.append(ExecutionInput(tasklet, input_file))
        return tasklets

    def on_execute(self):
        tasklet_token = sdk2.yav.Secret("sec-01fm366yft3rzqdwgza97km4p4", default_key="ci.token").value()
        executions = {}
        with arcadia_sdk.mount_arc_path(self.Parameters.arcadia_url) as arcadia_dir:
            if self.Parameters.tasklet_cli:
                tasklet_tool_command = [str(sdk2.ResourceData(self.Parameters.tasklet_cli).path)]
            else:
                tasklet_tool_command = [os.path.join(arcadia_dir, "ya"), "tool", "tasklet"]
            tasklet_cli = TaskletCli(tasklet_tool_command, self.Parameters.installation, tasklet_token)
            tasklets = []
            for namespace in self.Parameters.namespaces:
                for path in self.Parameters.tasklet_arcadia_paths:
                    tasklets.extend(self.get_examples_tasklets(tasklet_cli, arcadia_dir, namespace, path))

            logging.info("Try to run %s tasklets", len(tasklets))
            self.set_info("Executions")
            for tasklet_start_info in tasklets:
                tasklet = tasklet_start_info.tasklet
                input_file = tasklet_start_info.input_file
                cmd = ["run", tasklet.full_name(), input_file, "--account", "TASKLETS"]
                execution = Execution(tasklet_cli.command(cmd), tasklet)
                logging.info("Execution: %s. Tasklet: %s", execution.execution, tasklet.full_name())
                self.set_info(
                    "Execution: {}. Tasklet: {}".format(execution.link(), tasklet.full_name()), do_escape=False
                )
                executions[execution.id()] = Execution(execution, tasklet)

            completed_executions = {}

            while executions:
                time.sleep(60)
                for execution_id, execution in executions.items():
                    execution.execution = tasklet_cli.command(["execution", "get", execution_id])
                    if execution.finished():
                        completed_executions[execution_id] = execution
                for execution_id in completed_executions:
                    executions.pop(execution_id, None)

        executions_empty_results = []

        for execution_id, execution in completed_executions.items():
            if not execution.success():
                executions_empty_results.append(execution)

        if executions_empty_results:
            for execution in executions_empty_results:
                self.set_info(
                    "Failed execution: {}. Tasklet:".format(execution.link(), execution.tasklet.full_name()),
                    do_escape=False
                )
            raise common_errors.TaskError("Some executions failed.")

        logging.info("All executions are success")
