import logging
import os
import pathlib
import contextlib

from sandbox.common import errors
from sandbox.common import urls as common_urls
from sandbox.common.types import misc as ctm
from sandbox.projects.common import network

from sandbox import sdk2
from sandbox.sdk2.helpers.process import subprocess as sp

from sandbox.projects.sandbox.tasklets import resources as tasklet_resources


class Tasklet(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 4 << 10  # 4 GiB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):

        with sdk2.parameters.Group("Tasklet meta") as tasklet_spec:
            tasklet_executor_resource = sdk2.parameters.Resource("Executor resource", required=True)
            tasklet_to_execute_resource = sdk2.parameters.Resource("Tasklet resource", required=True)

        with sdk2.parameters.Group("Tasklet execution") as tasklet_execution:
            execution_id = sdk2.parameters.String("Tasklet execution ID", required=True)
            endpoint_address = sdk2.parameters.String("Address of tasklet grpc service")
            endpoint_set = sdk2.parameters.String("Endpoint set name of tasklet grpc service")

    @contextlib.contextmanager
    def _support_server_env(self):
        from tasklet.experimental.proto import sandbox_support_api_pb2_grpc as api_grpc
        from tasklet.experimental.proto import sandbox_support_api_pb2 as api_pb2
        import grpc
        from concurrent import futures

        class APIHandler(api_grpc.SandboxSupportAPIServicer):
            def GetResource(self, request: api_pb2.GetResourceRequest,
                            context: grpc.ServicerContext) -> api_pb2.GetResourceResponse:
                logging.info(f"Processing GetResource request. ResourceID: {request.resource_id}")
                try:
                    resource = sdk2.Resource[request.resource_id]
                    data = sdk2.ResourceData(resource)
                    logging.info("Resource %d downloaded to %s", request.resource_id, data.path)
                    return api_pb2.GetResourceResponse(path=str(data.path))
                except Exception as err:
                    logging.error(f"GetResource request failed. ResourceID: {request.resource_id}", exc_info=True)
                    context.set_code(grpc.StatusCode.INTERNAL)
                    context.set_details(repr(err))

            def __init__(self):
                pass

        server = grpc.server(futures.ThreadPoolExecutor(max_workers=10))
        api_grpc.add_SandboxSupportAPIServicer_to_server(
            APIHandler(), server
        )
        port = network.get_free_port()
        address = f'localhost:{port}'
        server.add_insecure_port(address)
        logging.info("starting support service on address: '%s'", address)
        server.start()
        try:
            yield address
        finally:
            logging.info("stopping support service")
            server.stop(None).wait()

    def on_execute(self):
        tasklet_executor_path = sdk2.ResourceData(self.Parameters.tasklet_executor_resource).path
        tasklet_to_execute_path = sdk2.ResourceData(self.Parameters.tasklet_to_execute_resource).path

        logs_resource = tasklet_resources.TaskletExecutionLogs(
            self,
            f"Logs of tasklet {self.Parameters.execution_id}",
            "tasklet_logs",
        )
        logs_data = sdk2.ResourceData(logs_resource)
        logs_data.path.mkdir(0o755, parents=True, exist_ok=True)

        logs_meta_link = common_urls.get_resource_link(logs_resource.id)
        logs_data_link = logs_resource.http_proxy
        self.set_info(
            f'Tasklet logs placed to <a href="{logs_meta_link}">resource #{logs_resource.id}</a>: '
            f'<a href="{logs_data_link}">{logs_data_link}</a>',
            do_escape=False,
        )

        executor_args = [
            str(tasklet_executor_path),
            "--execution-id", self.Parameters.execution_id,
            "--tasklet-path", str(tasklet_to_execute_path),
            "--logs-dir", str(logs_data.path),
        ]
        if self.Parameters.endpoint_address:
            executor_args.extend(["--endpoint-address", self.Parameters.endpoint_address])
        else:
            executor_args.extend(["--endpoint-set", self.Parameters.endpoint_set])

        env = os.environ.copy()
        env["SANDBOX_SESSION"] = self.server.task_token

        with self._support_server_env() as server_address:
            env["TASKLET_SUPPORT_API_URL"] = server_address
            with sdk2.helpers.ProcessLog(self, "tasklet-executor") as process_log:
                executor_process = sp.Popen(
                    executor_args,
                    env=env,
                    stdout=process_log.stdout,
                    stderr=process_log.stderr,
                )
                executor_process.communicate()

            if executor_process.returncode:
                raise errors.TaskError(f"Executor failed to execute tasklet {self.Parameters.execution_id}")
