import time
import logging

from sandbox import common

from sandbox import sdk2
from sandbox.sandboxsdk import environments


class WalleHostsChecker(sdk2.Task):
    """
    Find `ready` hosts with `CONFIG_REPLACES` keys and change their config to appropriate values.
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 512
        cores = 1
        environments = [environments.PipEnvironment("wall-e.api>=6.1.4,<7.0.0")]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 900  # 15 minutes
        walle_projects = sdk2.parameters.List("Projects in Wall-e", required=True)
        host_ready_timeout = sdk2.parameters.Integer("Host ready timeout", default=7200)

        with sdk2.parameters.Group("Walle OAuth token"):
            vault_item_owner = sdk2.parameters.String("Vault owner")
            vault_item_name = sdk2.parameters.String("Vault name")

    class Context(sdk2.Task.Context):
        updated_hosts = []

    CONFIG_REPLACES = {
        "sandbox-removelabels-search-ubuntu-16.04": "sandbox-search-ubuntu-16.04",
        "sandbox-hbf-removelabels-search-ubuntu-16.04": "sandbox-hbf-search-ubuntu-16.04",
    }

    @property
    def footer(self):
        return {
            "": [
                {"Host": host, "Old config": old, "New config": new}
                for host, old, new in self.Context.updated_hosts
            ]
        }

    @common.utils.singleton_property
    def walle_client(self):
        access_token = sdk2.Vault.data(self.Parameters.vault_item_owner, self.Parameters.vault_item_name)
        import walle_api
        return walle_api.WalleClient(access_token=access_token)

    def _get_ready_hosts_by_config(self, config):
        for project in self.Parameters.walle_projects:
            logging.info("Checking project: %s", project)
            for host in self.walle_client.iter_hosts(config=config, project=project):
                if "config" in host and host["status"] == "ready":
                    yield host["name"]

    def _last_audit_entry(self, host):
        for entry in self.walle_client.iter_audit_log(
            ["status", "status_time"], host_name=host, limit=1, reverse=True
        ):
            return entry
        return None

    def _replace_config(self, host, old, new):
        audit = self._last_audit_entry(host)

        if audit is None:
            logging.warning("There are no actions for %s, which is weird", host)
            return

        if audit["status"] != "completed":
            return

        if time.time() - audit["status_time"] < self.Parameters.host_ready_timeout:
            return

        self.walle_client.set_deploy_config(host, new)
        logging.info("Set deploy_config='%s' for %s", new, host)

        self.Context.updated_hosts.append((host, old, new))

    def on_execute(self):
        for old, new in self.CONFIG_REPLACES.iteritems():
            for host in self._get_ready_hosts_by_config(old):
                self._replace_config(host, old, new)
