# coding: utf-8

import logging
import datetime
import requests
import aniso8601
import itertools
import collections

import library.format

from sandbox import common, sdk2
from sandbox.sdk2 import parameters
from sandbox.common.types import task as ctt
from sandbox.common.types import client as ctc

from . import utils

TELEGRAPH_URL = "https://telegraph.yandex.net/api/v3"
STARTREK_URL = "https://st-api.yandex-team.ru/v2"

ROBOT_PHONE_NO = 19864  # robot-sandbox@'s phone number
SANDBOX_ABC_SERVICE_ID = 469  # slug: sandbox
SANDBOX_EMERGENCY_SCHEDULE_ID = 30627  # name: Sandbox Emergency
ABC_PROD_TVM_ID = 2012190  # according to https://wiki.yandex-team.ru/intranet/abc/api

MAX_TELEGRAM_OWNER_ABSENCE_INTERVAL = datetime.timedelta(days=90)

DATE_FORMAT = "%Y-%m-%d"
DATE_REPORT_FORMAT = DATE_FORMAT + " %A"
DESCRIPTION_SEPARATOR = "\n-----\n"


class WhoIsOnDutyToday(sdk2.Task):
    """
    Task to determine who is on duty. Create it with scheduler for proper previous state extracting.
    """

    class Parameters(sdk2.Parameters):
        dead_clients_msg_telegram_chat_id = sdk2.parameters.List(
            "Telegram chat identifiers for dead clients notification", default=[-1001075746847]
        )
        tvm_secret = sdk2.parameters.YavSecret(
            "Yav secret for TVM-secret (tvm.secret.2018182)",
            default="sec-01dzxzkd14km98wsmftebpp5f0",
            required=True
        )
        owner_last_visit = sdk2.parameters.List(
            "Telegram chat identifiers for owner last visit notification", default=[-1001075746847]
        )
        telegram_bot_token = sdk2.parameters.YavSecret(
            "Yav secret for yasandbot-telegram-token",
            default="sec-01g56fb670bc9azcrk1wfq8s2c"
        )

        with sdk2.parameters.Output:
            dead_clients_ids = sdk2.parameters.List("Dead clients ids list")
            assigned_duty_login = sdk2.parameters.String("Duty's Staff login")

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 4096
        disk_space = 15

        class Caches(sdk2.Requirements.Caches):
            pass  # no shared caches

    def _get_prev_task(self):
        prev_tasks = self.server.task.read(
            type=self.type.name,
            scheduler=[self.scheduler, -self.scheduler],
            limit=1,
            status=ctt.Status.SUCCESS,
            order="-id",
        )["items"]
        return prev_tasks[0] if prev_tasks else None

    def _notify_telegram_chat(self, chat_id, message):
        try:
            bot_token = self.Parameters.telegram_bot_token.data()["token"]
            bot = common.telegram.TelegramBot(bot_token)
            bot.send_message(chat_id, message, common.telegram.ParseMode.HTML)
        except Exception:
            err_message = "Failed to send chat notification"
            logging.exception(err_message, exc_info=True)
            raise common.errors.TemporaryError(err_message)

    def _notify(self, new_dead_clients, last_owner_visit):
        telegram_account = None
        grouped_clients = self._group_clients_by_tags(
            new_dead_clients, tag_groups=[ctc.Tag.Group.DENSITY, ctc.Tag.Group.PURPOSE, ctc.Tag.Group.SERVICE]
        )
        dead_clients_message = "New dead clients:\n{}\nTotal: {}".format(
            "\n".join([
                "{} ({}): {}".format(group, len(clients), library.format.formatHosts(clients))
                for group, clients in grouped_clients.items()
            ]),
            len(new_dead_clients)
        )
        dead_clients_message = '<strong>{}</strong>\n{}\nTask: <a href="{}">{}</a>'.format(
            datetime.datetime.today().strftime(DATE_REPORT_FORMAT),
            dead_clients_message,
            common.utils.get_task_link(self.id),
            self.id,
        )
        logging.info("New dead clients: %s", grouped_clients)
        for chat_id in self.Parameters.dead_clients_msg_telegram_chat_id:
            self._notify_telegram_chat(chat_id, dead_clients_message)

        last_owner_visit_message = "Telegram bot owner last visit: {}".format(
            common.format.dt2str(last_owner_visit)
        )
        owner_absence_notification = None
        if datetime.datetime.utcnow() - MAX_TELEGRAM_OWNER_ABSENCE_INTERVAL > last_owner_visit:
            owner_absence_notification = "{} <strong>Warning: telegram owner expired</strong>".format(
                telegram_account if telegram_account else ""
            )
        for chat_id in self.Parameters.owner_last_visit:
            self._notify_telegram_chat(chat_id, last_owner_visit_message)
            if owner_absence_notification:
                self._notify_telegram_chat(chat_id, owner_absence_notification)

    def _get_prev_dead_clients_ids(self):
        if self.scheduler:
            prev_task = self._get_prev_task()
            if prev_task:
                return set(prev_task["output_parameters"].get("dead_clients_ids", []))
        return set()

    def _get_dead_clients(self, chunk_size=3000):
        result = []
        for offset in itertools.count(0, chunk_size):
            clients = self.server.client.read(alive=False, tags="~MAINTENANCE", offset=offset, limit=chunk_size)
            if not clients["items"]:
                break
            result.extend(clients["items"])
        return result

    @staticmethod
    def _filter_client_density_tags(client_tags, ordered_density_tags):
        result = []
        client_density_tags = []

        for tag in client_tags:
            if tag in ctc.Tag.Group.DENSITY:
                client_density_tags.append(tag)
            else:
                result.append(tag)

        if client_density_tags:
            result.append(max(client_density_tags, key=lambda tag: ordered_density_tags.index(tag)))

        return result

    @classmethod
    def _group_clients_by_tags(cls, clients, tag_groups):
        expanded_tag_group = ctc.Tag.Group.expand(*tag_groups)
        client_groups = collections.defaultdict(list)
        ordered_density_tags = filter(lambda tag: tag in ctc.Tag.Group.DENSITY, ctc.Tag)

        for client in clients:
            tags = cls._filter_client_density_tags(client["tags"], ordered_density_tags)
            for tag in tags:
                if ctc.Tag[tag] in expanded_tag_group:
                    client_groups[tag].append(client["fqdn"])

        return client_groups

    def __set_assigned_duty_login(self):
        # this package is not listed in "environments" since it's already present in /skynet/python
        # moreover, due to some reason the same package downloaded from pip does not fit the unicode length
        import ticket_parser2.api.v1 as ticket_parser
        tvm_ticket_id = 2018182
        tvm_token = self.Parameters.tvm_secret.data()["client_secret"]
        service_ticket = utils.APIClient.acquire_service_ticket(
            ticket_parser,
            tvm_ticket_id,
            tvm_token,
            destinations={"abc": ABC_PROD_TVM_ID}
        )
        duty_api = utils.DutyAPI(service_ticket=service_ticket)
        current_duties = duty_api.get_current_duty(
            service=SANDBOX_ABC_SERVICE_ID, schedule=SANDBOX_EMERGENCY_SCHEDULE_ID
        )
        if len(current_duties) > 1 and any(current_duties[0].login != duty.login for duty in current_duties):
            raise SystemExit("Got several on_duty results with different users. Please, specify schedule")
        self.Parameters.assigned_duty_login = current_duties[0].login

    def on_execute(self):
        self.Parameters.description = self.Parameters.description.split(DESCRIPTION_SEPARATOR)[0]

        try:
            prev_dead_clients_ids = self._get_prev_dead_clients_ids()
            dead_clients = self._get_dead_clients()
            new_dead_clients = filter(lambda client: client["id"] not in prev_dead_clients_ids, dead_clients)
            last_owner_visit = (self.server >> common.rest.Client.PLAINTEXT).service.telegram.owner_last_visit.read()
            last_owner_visit = aniso8601.parse_datetime(last_owner_visit).replace(tzinfo=None)

            self.__set_assigned_duty_login()
            self.Parameters.dead_clients_ids = [client["id"] for client in dead_clients]
            self._notify(new_dead_clients, last_owner_visit)

        except requests.RequestException as error:
            logging.exception("Error occurred: %r", error)
            if error.response and error.response.status_code < 500:
                raise
            raise common.errors.TemporaryError("Requests error occurred: {}. See logs for details.".format(error))
