# coding: utf-8

import json
import logging
import datetime
import requests
import xml.etree
import collections

from sandbox import common
import sandbox.common.types.misc as ctm

DayDescription = collections.namedtuple("DayDescription", ["date", "is_holiday", "day_type", "holiday_name"])


class APIClient(object):

    def __init__(self, oauth_token=None, service_ticket=None, max_tries=3, timeout=15):
        self._oauth_token = oauth_token
        self._service_ticket = service_ticket
        self._max_tries = max_tries
        self._timeout = timeout

    @staticmethod
    def _get_tvm_client(tp2, tvm_ticket_id, tvm_token, destinations):
        """
        :param tp2: Module Task parser (library.ticket_parser2.api.v1) is expected to be imported in
            outer code to use this function
        :param tvm_ticket_id: castable to int tvm id
        :param tvm-token: str token to connect to service
        :param destinations: dictionary of destination name :string to destination id :int
        :return: TvmClient
        """

        client = tp2.TvmClient(tp2.TvmApiClientSettings(
            self_client_id=int(tvm_ticket_id),
            enable_service_ticket_checking=True,
            self_secret=tvm_token,
            dsts=destinations,
        ))

        return client

    @classmethod
    def acquire_service_ticket(cls, tp2, tvm_ticket_id, tvm_token, destinations):
        """
        :param tp2: module Task parser (library.ticket_parser2.api.v1) is expected to be imported in
            outer code to use this function
        :param tvm_ticket_id: castable to int tvm id
        :param tvm_token: tvm token from the yav
        :param destinations: dictionary of destination name :string to destination id :int
        :return: Dictionary of destination name to tvm_ticket
        :rtype: dict
        """

        tvm_client = APIClient._get_tvm_client(tp2, tvm_ticket_id, tvm_token, destinations)
        service_ticket = tvm_client.get_service_ticket_for(list(destinations)[0])

        return service_ticket

    def _retryable_get(self, *args, **kwargs):
        kwargs.setdefault("allow_redirects", True)
        kwargs.setdefault("timeout", self._timeout)
        # FIXME: SSLError: hostname 'staff-api.yandex-team.ru' doesn't match either of
        # FIXME: 'h.yandex.net', 'h.yandex-team.ru'
        kwargs.setdefault("verify", False)
        response = None
        tries_left = self._max_tries
        while tries_left > 0:
            logging.debug("Retrieving URL via _retryable_get(%r, %r)", args, kwargs)
            response = requests.get(*args, **kwargs)
            if response.status_code < 500:  # not server error
                logging.debug(
                    "URL retrieving succeeded (status: %d, content: %r)",
                    response.status_code, response.text
                )
                return response
            tries_left -= 0
        if response:
            response.raise_for_status()
        raise requests.HTTPError("Unknown error")

    def _get_oauth_headers(self):
        return {"Authorization": "OAuth {}".format(self._oauth_token)} if self._oauth_token else {}

    def _get_service_ticket_header(self):
        return {ctm.HTTPHeader.SERVICE_TICKET: self._service_ticket}


class Country(common.utils.Enum):
    RUSSIA = 225


class CalendarAPI(APIClient):

    BASE_URL_OAUTH = "https://calendar.yandex-team.ru"
    BASE_URL = "https://calendar-api.tools.yandex.net/internal"

    @staticmethod
    def _parse_holidays_json(content):
        holidays_container = json.loads(content)
        logging.info(holidays_container.__str__())
        holidays = {}
        for day in holidays_container["holidays"]:
            holidays[day["date"]] = DayDescription(
                date=day["date"],
                is_holiday=day["type"] in ("holiday", "weekend"),
                day_type=day["type"],
                holiday_name=day["name"] if "name" in day else "unnamed",
            )
        return holidays

    @staticmethod
    def _parse_holidays(text):
        root = xml.etree.ElementTree.fromstring(text.encode("utf-8"))
        days = root.find("get-holidays").find("days")
        holidays = {}
        for day in days:
            holidays[day.attrib["date"]] = DayDescription(
                date=day.attrib["date"],
                is_holiday=day.attrib["is-holiday"],
                day_type=day.attrib["day-type"],
                holiday_name=day.attrib.get("holiday-name", ""),
            )
        return holidays

    @staticmethod
    def _to_str_date(date):
        if isinstance(date, datetime.datetime):
            date = date.date()
        if isinstance(date, datetime.date):
            date = date.isoformat()
        assert isinstance(date, basestring)
        return date

    def _get_holidays_tvm_api(self, start_date, end_date, country_id=Country.RUSSIA):
        """
        :rtype: dict[str, DayDescription]
        """

        # it's expected dates to be without the time specification
        start_date, end_date = str(start_date).split()[0], str(end_date).split()[0]
        response = self._retryable_get(
            self.BASE_URL + "/get-holidays",
            params={
                "from": start_date,
                "to": end_date,
                "for": country_id,
                "outMode": "holidays"
            },
            headers=self._get_service_ticket_header()
        )

        response.raise_for_status()
        holidays = self._parse_holidays_json(response.text)
        logging.debug("Parsed holidays for country %s: %r", country_id, holidays)
        return holidays

    # todo: this method should be replaced with self._get_holidays_new_api as soon as
    # todo: there will not be any tasks using oauth with the calendar api
    def get_holidays(self, start_date, end_date, country_id=Country.RUSSIA):
        """
        :rtype: dict[str, DayDescription]
        """
        if self._service_ticket:
            return self._get_holidays_tvm_api(start_date, end_date, country_id=country_id)

        start_date = self._to_str_date(start_date)
        end_date = self._to_str_date(end_date)
        response = self._retryable_get(
            self.BASE_URL_OAUTH + "/export/holidays.xml",
            params={
                "start_date": start_date,
                "end_date": end_date,
                "country_id": str(country_id),
                "out_mode": "holidays",
                "who_am_i": "sandbox",
            },
            headers=self._get_oauth_headers(),
        )
        if response.status_code == requests.codes.ok:
            holidays = self._parse_holidays(response.text)
            logging.debug("Parsed holidays for country %s: %r", country_id, holidays)
            return holidays
        response.raise_for_status()


class Duty(object):
    def __init__(self, data):
        self.login = data["person"]["login"]


class DutyAPI(APIClient):
    BASE_URL = "https://abc-back.yandex-team.ru/api/v4"
    ON_DUTY_URL = BASE_URL + "/duty/shifts/"

    def get_current_duty(self, service, schedule=None):
        date_from = date_to = datetime.datetime.utcnow().date()
        params = dict(
            date_from=date_from,
            date_to=date_to,
            with_watcher=1,
            service=service,
            schedule=schedule,
        )
        response = self._retryable_get(self.ON_DUTY_URL, params=params, headers=self._get_service_ticket_header())
        if response.status_code == requests.codes.ok:
            logging.debug("response.json()['results'][0]: {}".format(response.json()["results"][0]))
            logging.debug("type = {0}".format(type(response.json()["results"][0])))
            return [Duty(response.json()["results"][0])]
        response.raise_for_status()
