# -*- coding: utf-8 -*-

import datetime
import logging
import os
import re
import subprocess
import traceback


from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog
from sandbox.common.errors import TaskFailure
from sandbox.common.types import misc as ctm
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.resource_types import CALC_METRICS_DAEMON_EXECUTABLE
from sandbox.projects.sandbox_ci.managers.statface import StatfaceManager
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.environments import ArcEnvironment
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.sandbox_ci.counters_metrics.projects import PROJECTS

def get_version_in_package_json(project_path):
    package_json_path = os.path.join(project_path, 'package.json')

    with open(package_json_path, 'r') as package_json_read:
        content = package_json_read.read()
        return re.sub('^.*--calc-metrics-daemon-version (stable-[0-9]+-[0-9]+.[0-9]+).*$', '\g<1>', content, flags=re.S)


class MonitorMetricsDaemonInProjects(sdk2.Task):
    '''
    Задача предназначена для мониторинга версии демона метрик в сервисах в аркадии с тестами на archon.

    Задача запускается по расписанию (каждый день) и записывает в Stat текущую версию в проекте и отставание версии в проекте в днях.
    '''

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        environments = [PipEnvironment('python-statface-client', version="0.154.0", custom_parameters=["requests==2.18.4"])]

    class Parameters(sdk2.Parameters):
        pass

    def on_execute(self):
        self.secret = sdk2.yav.Secret('sec-01fpd13dp7r9229g2s4a81zgdq').data()

        data = []
        old_daemons = []

        with arcadia_sdk.mount_arc_path('arcadia-arc:/#trunk') as arcadia_root:
            for project_name in PROJECTS:
                project = PROJECTS[project_name]
                logging.info('project {}: {}'.format(project_name, project))
                project_path = os.path.join(arcadia_root, project['path'])

                current_version = get_version_in_package_json(project_path)
                logging.info('project {}: current version {}'.format(project_name, current_version))

                daemon = CALC_METRICS_DAEMON_EXECUTABLE.find(attrs=dict(resource_version=current_version,), state=ctr.State.READY).first()
                logging.info('project {}: daemon {} [created {}]'.format(project_name, daemon.id, daemon.created))

                lag = datetime.datetime.now(daemon.created.tzinfo) - daemon.created
                yesterday = datetime.datetime.now() - datetime.timedelta(days=1)

                data.append(dict(
                    fielddate=yesterday.strftime('%Y-%m-%d'),
                    project=project_name,
                    project_version=current_version,
                    lag_in_days=lag.days
                ))

                if lag.days > 14 and project['alert_on_old_version']:
                    old_daemons.append(dict(project=project_name, version=current_version, lag=lag.days))

        logging.info('data for upload: {}'.format(data))
        logging.info('old version of daemon: {}'.format(old_daemons))

        # catch statface errors, see https://st.yandex-team.ru/FEI-5011
        import socket
        from statface_client import StatfaceClient, StatfaceClientError
        from requests.exceptions import HTTPError, ConnectionError
        try:
            report_path = 'Yandex/Infra/CountersMetrics/ProjectAndProductionVersionsDiff'

            stat_oauth_token = self.secret['stat']
            statface = StatfaceClient(oauth_token=stat_oauth_token, host='upload.stat.yandex-team.ru')

            logging.info('statface client', statface)

            report = statface.get_report(report_path)
            logging.info('statface report', report)
            report.upload_data(scale=u'd', data=data)

        except (StatfaceClientError, HTTPError, ConnectionError, socket.error):
            raise TaskFailure('Не удалось загрузить данные в Stat:\n{}'.format(traceback.format_exc()))

        if len(old_daemons) > 0:
            raise TaskFailure('В следующих проектах устаревший демон метрик:\n{}'.format(old_daemons))
