# -*- coding: utf-8 -*-

import logging
import os
import re
import subprocess


from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog
from sandbox.common.errors import TaskFailure
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.sandbox_ci.utils import poll, arcanum
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.environments import ArcEnvironment
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.sandbox_ci.counters_metrics.projects import PROJECTS

FILTER_TEMPLATE = '- ((https://st.yandex-team.ru/issues/?_q=Queue%3A+{}+Tags%3A+metrics-daemon-update++%22Sort+by%22%3A+Created+DESC Все задачи на обновление демона метрик))'

POLLING_PR_INFO_INTERVAL = 10
POLLING_PR_INFO_TIMEOUT = 300


class UpdateMetricsDaemonInProjects(sdk2.Task):
    '''
    Задача предназначена для обновления версии демона метрик в сервисах в аркадии с тестами на archon.

    Обновление версии включает в себя:
    - создание соответствующей задачи в трекере
    - создание ревью-реквеста, привязанного к данной задаче

    Задача используется в релизной машине кода метрик.
    '''

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        environments = [PipEnvironment('startrek_client', use_wheel=True)]

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Ресурсы calc_metrics_daemon') as resources:
            linux_resource = sdk2.parameters.Resource('Ресурс calc_metrics_daemon для Linux', required=True)
            darwin_resource = sdk2.parameters.Resource('Ресурс calc_metrics_daemon для MacOS', required=True)

        with sdk2.parameters.CheckGroup('Проекты') as projects:
            for project in PROJECTS:
                projects.values[project] = projects.Value(project, checked=True)

    tracker_client = None
    arcanum_client = None
    version = None

    def on_enqueue(self):
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=map(lambda p: ctt.Semaphores.Acquire(
                name='sandbox_ci_update_metrics_daemon_in_projects_{}'.format(p),
                capacity=1
            ), self.Parameters.projects)
        )

    def on_execute(self):
        self.secret = sdk2.yav.Secret('sec-01fpd13dp7r9229g2s4a81zgdq').data()
        self.version = self.Parameters.linux_resource.resource_version

        for project_name in self.Parameters.projects:
            project = PROJECTS[project_name]
            logging.info('project {}: {}'.format(project_name, project))

            with arcadia_sdk.mount_arc_path('arcadia-arc:/#trunk', arc_oauth_token=self.secret['arc']) as arcadia_root:
                project_path = os.path.join(arcadia_root, project['path'])

                current_version = self.update_version_in_package_json(arcadia_root, project_path)

                if not current_version:
                    logging.info('project {}: already uses new version {}', project_name, self.version)
                    continue

                issue = self.create_issue(current_version, project)
                logging.info('project {}: created issue {} - {}'.format(project_name, issue.key, issue.self))

                pr = self.create_review_request(project_name, project_path, issue)
                logging.info('project {}: created RR https://a.yandex-team.ru/review/{}'.format(project_name, pr))

                self.add_review_request_link_to_issue(issue, pr)
                self.set_status_to_issue_before_merging(issue, project['transitions'])
                self.merge(pr)

    def update_version_in_package_json(self, arcadia_root, project_path):
        current_version = None
        package_json_path = os.path.join(project_path, 'package.json')

        new_content = None
        with open(package_json_path, 'r') as package_json_read:
            content = package_json_read.read()

            current_version = re.sub('^.*--calc-metrics-daemon-version (stable-[0-9]+-[0-9]+.[0-9]+).*$', '\g<1>', content, flags=re.S)

            if self.version == current_version:
                return

            new_content = content.replace(current_version, self.version)

        with open(package_json_path, 'w') as package_json_write:
            package_json_write.write(new_content)

        return current_version

    def create_review_request(self, project_name, project_path, issue):
        commit_msg = 'calc-metrics-daemon: {}'.format(self.version)
        with ProcessLog(self, logger='arc') as arc_log:
            arc = ArcEnvironment().prepare()
            env = {
                'ARC_TOKEN': self.secret['arc'],
                'ARC_USER': self.secret['login']
            }
            branch_name = '{}_{}'.format(issue.key, project_name)
            summary = '{}: {}'.format(issue.key, issue.summary)

            subprocess.Popen('{} co -b {}'.format(arc, branch_name), shell=True, stdout=arc_log.stdout, stderr=arc_log.stderr, cwd=project_path, env=env).wait()
            subprocess.Popen('{} add .'.format(arc), shell=True, stdout=arc_log.stdout, stderr=arc_log.stderr, cwd=project_path, env=env).wait()
            subprocess.Popen('{} ci -m "{}"'.format(arc, commit_msg), shell=True, stdout=arc_log.stdout, stderr=arc_log.stderr, cwd=project_path, env=env).wait()
            subprocess.Popen('{} pr create --push --publish -m "{}"'.format(arc, summary), shell=True, stdout=arc_log.stdout, stderr=arc_log.stderr, cwd=project_path, env=env).wait()

            out = arc_log.stdout.path.read_text()
            pr = parse_pr(out)

            if not pr:
                logging.info('arc.out.log на момент получения id pull-request\'a:\n{}'.format(out))
                raise TaskFailure('В выводе команды arc pr create нет номера PR, что-то пошло не так. Напишите в https://wiki.yandex-team.ru/infraduty/')

            return pr

    def get_tracker_client(self):
        if self.tracker_client:
            return self.tracker_client

        from startrek_client import Startrek
        startrek_token = self.secret['startrek']
        self.tracker_client = Startrek(useragent='startrek_python_client', base_url='https://st-api.yandex-team.ru', token=startrek_token)
        return self.tracker_client

    def create_issue(self, current_version, project):
        client = self.get_tracker_client()

        task_summary = u'Обновление демона метрик: {} -> {}'.format(current_version, self.version)
        task_filter = FILTER_TEMPLATE.format(project['queue'])
        description = '{}\n{}\n'.format(project['description'], task_filter)

        issue = client.issues.create(
            queue=project['queue'],
            summary=task_summary,
            description=description,
            tags=project['tags'],
        )

        return issue

    def add_review_request_link_to_issue(self, issue, pr):
        description = '{}- Ревью-реквест: https://a.yandex-team.ru/review/{}'.format(issue.description, pr)
        issue.update(description=description)

    def set_status_to_issue_before_merging(self, issue, transitions):
        for status in transitions:
            issue.transitions[status].execute()

    def merge(self, pr):
        client = self.get_arcanum_client()

        try:
            poll.wait_for(
                dict(
                    ensure_pr_is_ready_for_merge_comment=lambda: self.ensure_pr_is_ready_for_merge_comment(pr)
                ),
                interval=POLLING_PR_INFO_INTERVAL,
                timeout=POLLING_PR_INFO_TIMEOUT
            )

            client.send_comment(pr, '/merge')
        except Exception as e:
            logging.info('Не удалось автоматически влить ревью-реквест:\n{}'.format(e))
            raise TaskFailure('Ошибка при отправке комментария /merge')

    def get_arcanum_client(self):
        if (self.arcanum_client):
            return self.arcanum_client

        arcanum_api_token = self.secret['arc']
        self.arcanum_client = arcanum.ArcanumApi(token=arcanum_api_token)

        return self.arcanum_client

    def ensure_pr_is_ready_for_merge_comment(self, pr):
        client = self.get_arcanum_client()
        json_body = client.get_pr_info(pr, 'checks(system,type,status)')

        return self.is_pr_published(json_body['checks'])

    def is_pr_published(self, checks):
        published_checks = [check for check in checks if self.is_published_check(check)]

        return len(published_checks) > 0 and published_checks[0]['status'] == 'success'

    def is_published_check(self, check):
        return check['system'] == 'arcanum' and check['type'] == 'published'


def parse_pr(out):
    matches = re.search('^.*https://a.yandex-team.ru/review/(\d+)', out, re.MULTILINE)
    if matches:
        return matches.group(1)
