# -*- coding: utf-8 -*-

import logging

from sandbox.common.types import task as ctt
from sandbox import sdk2

from sandbox.projects.sandbox_ci.dobby.base import InfratestDobbyBase
from sandbox.projects.sandbox_ci.dobby.simple import InfratestDobby

PROJECTS = ['web4', 'fiji', 'nerpa', 'frontend', 'frozen']


def get_subtask_description(project, tools):
    tools_chunks = [
        '{} to {}'.format(tool, ref)
        for tool, ref in tools.iteritems()
    ]
    return 'project: {}\nupdate tools:\n{}'.format(project, '\n'.join(tools_chunks))


class InfratestDobbyIntegral(InfratestDobbyBase):
    """
        Интеграционная задача для запуска INFRATEST_DOBBY для указанных проектов
    """

    class Parameters(InfratestDobbyBase.Parameters):
        description = 'Интеграционная задача для запуска dobby make для выбранных проектов'

        with sdk2.parameters.Group('Dobby Parameters'):
            tools = sdk2.parameters.Dict('Tools to update and their versions', required=True)
            task = sdk2.parameters.String('Task ID')
            with sdk2.parameters.String('Package file') as package_file:
                package_file.values['package.json'] = package_file.Value('update dependencies in package.json', default=True)
                package_file.values['bower.json'] = package_file.Value('update dependencies in bower.json')
            merge = sdk2.parameters.Bool('Add "/merge" comment')
            no_verify = sdk2.parameters.Bool('Add --no-verify flag', default=True)

            with sdk2.parameters.CheckGroup("Projects") as projects:
                for project in PROJECTS:
                    projects.values[project] = projects.Value(project, checked=True)

    def on_execute(self):
        with self.memoize_stage.mk_subtasks(commit_on_entrance=False):
            subtasks = []

            for project in self.Parameters.projects:
                subtasks.append(self.create_subtask(
                    self.Parameters.tools,
                    project,
                    self.Parameters.task,
                    self.Parameters.merge,
                    self.Parameters.package_file,
                    no_verify=self.Parameters.no_verify,
                ))

            raise sdk2.WaitTask(
                tasks=self.meta.start_subtasks(subtasks),
                statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )
        with self.memoize_stage.process():
            self.Context.report_resources = self.meta.get_subtasks_artifacts_ids()

    def create_subtask(self, tools, project, task, merge, package_file, **kwargs):
        logging.info('create subtask for: tools=%s project=%s task=%s', tools, project, task)

        return self.meta.create_subtask(
            task_type=InfratestDobby,
            tools=tools,
            project=project,
            task=task,
            merge=merge,
            package_file=package_file,
            description=get_subtask_description(project, tools),
            **kwargs
        )
