# -*- coding: utf-8 -*-

from sandbox import sdk2
import logging

from sandbox.projects.common.vcs.arc import Arc
from sandbox.common.types import task as ctt

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper, Node
from sandbox.projects.sandbox_ci.dobby.base import InfratestDobbyBase


class InfratestDobby(InfratestDobbyBase):
    """
        Таск для запуска dobby make в рамках одного проекта
        Подробнее про dobby здесь: https://gitlab.yandex-team.ru/search-interfaces/dobby
    """
    class Parameters(InfratestDobbyBase.Parameters):
        description = 'dobby make'

        with sdk2.parameters.Group('Dobby Parameters'):
            tools = sdk2.parameters.Dict('Tools to update and their versions', required=True)
            project = sdk2.parameters.String('Project', required=True)
            task = sdk2.parameters.String('Task ID')
            merge = sdk2.parameters.Bool('Add "/merge" comment')
            package_file = sdk2.parameters.String('Update dependencies in')
            no_verify = sdk2.parameters.Bool('Add --no-verify flag', default=True)

    lifecycle_steps = {
        'npm_install': 'npm i --production --registry=http://npm.yandex-team.ru @yandex-int/dobby',
        'dobby_config_startrek': '{dobby_path} config startrek.token {startrek_token}',
        'dobby_config_aracnum': '{dobby_path} config arcanum.token {arcanum_token}',
        'dobby_config_username': '{dobby_path} config username {username}',
        'dobby_config_arc_mount': '{dobby_path} config arc.mount {arc_mount}',
        'dobby_make': 'DEBUG=* {dobby_path} make {tools_opts} -s {project} -ci {options}',
    }

    """
        Наличие свойста "project_name" и методов "project_dir" "working_path" "project_path"
        обусловлено желанием по максимуму использовать существующую функциональность
        из sandbox.projects.sandbox_ci.managers. Но код модулей managers предполагает использование
        совместно с sandbox.projects.sandbox_ci.task.BaseTask. И чтобы реиспользовать код пришлось
        добавить и переопределить вышеуказанные свойства и методы в данной задаче.
    """
    @property
    def project_name(self):
        return ''

    @property
    def project_dir(self):
        return self.working_path()

    def working_path(self, *args):
        return self.path(*args)

    def project_path(self, *args):
        return self.project_dir.joinpath(*args)

    @property
    def report_description(self):
        return 'dobby report'

    def on_prepare(self):
        tools_chunks = [
            '--tool {} --tool-version {}'.format(tool, ref)
            for tool, ref in self.Parameters.tools.iteritems()
        ]
        tools_opts = ' '.join(tools_chunks)

        options = []
        if self.Parameters.task:
            options.append('-t {}'.format(self.Parameters.task))
        if self.Parameters.merge:
            options.append('--merge')
        if (self.Parameters.package_file == 'bower.json'):
            options.append('--package-file {}'.format(self.Parameters.package_file))
        if self.Parameters.no_verify:
            options.append('--no-verify')

        self.lifecycle.update_vars(
            dobby_path='node_modules/.bin/dobby',
            startrek_token=self.vault.read('robot-serp-bot_startrek_token'),
            arcanum_token=self.vault.read('env.ARCANUM_OAUTH_TOKEN'),
            username='robot-serp-bot',
            tools_opts=tools_opts,
            project=self.Parameters.project,
            options=' '.join(options),
        )

    def on_execute(self):
        logging.info('Dobby start:')

        for tool, ref in self.Parameters.tools.iteritems():
            logging.info('Tool: {}'.format(tool))
            logging.info('Reference: {}'.format(ref))
        logging.info('Project: {}'.format(self.Parameters.project))
        logging.info('Task ID: {}'.format(self.Parameters.task))

        arc_token = self.vault.read('env.ARC_TOKEN')
        with GitRetryWrapper(), Node(self.Parameters.node_js_version), self.vault.ssh_key(), Arc(arc_oauth_token=arc_token).mount_path(None, None, fetch_all=False) as arc_mount_dir:
            self.lifecycle.update_vars(arc_mount=arc_mount_dir)
            self.lifecycle('npm_install')
            self.lifecycle('dobby_config_startrek')
            self.lifecycle('dobby_config_aracnum')
            self.lifecycle('dobby_config_username')
            self.lifecycle('dobby_config_arc_mount')
            self.lifecycle('dobby_make')

        logging.info('Dobby report:')
        self._create_report(self.Parameters.project)

    def _create_report(self, project):
        self.artifacts.create_report(
            resource_path='report',
            type=project,
            status=ctt.Status.SUCCESS,
            add_to_context=True,
            public=True,
            root_path='index.html'
        )
