# -*- coding: utf-8 -*-
import logging
import tarfile
import StringIO

from sandbox import sdk2
from sandbox.common.errors import ResourceNotFound
from sandbox.common.types import resource as ctr


class Resource(object):
    @staticmethod
    def find(resource_type, **attrs):
        """
        Ищет готовый ресурс указанного типа по заданным аттрибутам.

        :param resource_type: Тип ресурса
        :type resource_type: str
        :param attrs: Аттрибуты для поиска
        :type attrs: dict
        :rtype: sdk2.Resource
        """
        params = dict(
            type=resource_type,
            state=ctr.State.READY,
            attrs=attrs,
        )

        logging.debug('Searching resource with params: {}'.format(params))

        resource = sdk2.Resource.find(**params).order(-sdk2.Resource.id).first()

        if resource is None:
            raise ResourceNotFound('Unable to find resource with params: {}'.format(params))

        logging.debug('Resource {resource} with params: {params}'.format(
            params=params,
            resource=resource,
        ))

        return resource

    @staticmethod
    def unpack(resource, destination):
        """
        Распакует архив в указанную директорию.

        :param resource: Ресурс
        :type resource: sdk2.Resource
        :param destination: Путь распаковки
        :type destination: str
        """
        source = str(sdk2.ResourceData(resource).path)

        logging.debug('Unpacking {resource} from "{source}" to "{destination}"'.format(
            resource=resource,
            source=source,
            destination=destination,
        ))

        with tarfile.open(source, mode='r:*') as tar:
            tar.extractall(destination)

    @staticmethod
    def create(task, data, resource_type='SANDBOX_CI_ARTIFACT', filename='data.json', archive='data.tar.gz'):
        """
        Сохраняет данные в файл, архивирует и из архива варит ресурс указанного типа.

        :param task: Таск
        :type task: sdk2.Task
        :param data: Данные
        :type data: str
        :param resource_type: Тип ресурса
        :type resource_type: str
        :param data: Тест-сьюты
        :type data: dict
        :param filename: Название файла
        :type filename: str
        :param archive: Название архива
        :type archive: str
        :rtype: sdk2.Resource
        """
        with tarfile.open(archive, 'w:gz') as tar:
            tarinfo = tarfile.TarInfo(filename)
            tarinfo.size = len(data)

            logging.debug('Adding to {archive}: {data}'.format(
                archive=archive,
                data=data,
            ))

            tar.addfile(tarinfo, StringIO.StringIO(data))

        resource = sdk2.Resource[resource_type](
            task=task,
            description='{} with data'.format(archive),
            path=archive,
        )

        logging.debug('Resource with {archive}: {resource}'.format(
            archive=archive,
            resource=resource,
        ))

        sdk2.ResourceData(resource).ready()

        return resource
