# -*- coding: utf-8 -*-
import logging
import json

from operator import attrgetter
from itertools import groupby, islice

from sandbox.common.types import misc as ctm
from sandbox.common.types import resource as ctr
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.utils import flow
from sandbox.projects.sandbox_ci.task import ManagersTaskMixin
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement
from sandbox.projects.sandbox_ci.infrastats.date_util import fill_date_variants, convert_to_date
from sandbox.agentr.errors import ResourceNotAvailable

ITEMS_PER_PAGE = 1000
MAX_PARALLEL = 20
STATFACE_REPORT_PREFIX = 'Yandex.Productivity/infrastats/gemini-auto'


def load_all(request):
    limited_req = request.limit(ITEMS_PER_PAGE)
    first_req = limited_req.offset(0)
    res = list(first_req)

    count = first_req.count
    for offset in range(ITEMS_PER_PAGE, count, ITEMS_PER_PAGE):
        res.extend(limited_req.offset(offset))
    return res


class InfratestInfrastatsGeminiAuto(TasksResourceRequirement, ManagersTaskMixin, sdk2.Task):
    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = 3 * 1024
        environments = (PipEnvironment('python-statface-client', custom_parameters=["requests==2.18.4"]),)

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Filter parameters') as filter_block:
            with sdk2.parameters.String('Date From', required=True) as date_from:
                date_from.values['today'] = date_from.Value('Start of Today', default=True)
                fill_date_variants(date_from)

            with sdk2.parameters.String('Date To', required=True) as date_to:
                date_to.values['now'] = date_to.Value('Now', default=True)
                fill_date_variants(date_to)

            with sdk2.parameters.RadioGroup('Project') as project:
                project.values['web4'] = project.Value('web4', default=True)
                project.values['fiji'] = 'fiji'
                project.values['nerpa'] = 'nerpa'
                project.values['turbo'] = 'turbo'

            with sdk2.parameters.String('Service (fiji only)') as service:
                service.values[''] = service.Value('---', default=True)
                service.values['images'] = service.Value('Images')
                service.values['video'] = service.Value('Video')

            with sdk2.parameters.CheckGroup('Platforms') as platforms:
                platforms.values['desktop'] = platforms.Value('desktop', checked=True)
                platforms.values['touch-phone'] = platforms.Value('touch-phone', checked=True)
                platforms.values['touch-pad'] = platforms.Value('touch-pad', checked=True)
                platforms.values['tv'] = platforms.Value('tv', checked=False)

    class Context(sdk2.Context):
        report_resources = []

    @sdk2.header()
    def header(self):
        report = {}

        return report

    """
    Наличие свойства "project_name" и методов "project_dir", "working_path", "project_path"
    обусловлено желанием по максимуму использовать существующую функциональность
    из sandbox.projects.sandbox_ci.managers. Но код модулей managers предполагает использование
    совместно с sandbox.projects.sandbox_ci.task.BaseTask. И чтобы реиспользовать код пришлось
    добавить и переопределить вышеуказанные свойства и методы в данной задаче.
    """
    @property
    def project_name(self):
        return ''

    @property
    def project_dir(self):
        return self.working_path()

    def working_path(self, *args):
        return self.path(*args)

    def project_path(self, *args):
        return self.project_dir.joinpath(*args)

    def on_execute(self):
        logging.info('Starting to execute infrastats gemini auto task')

        resources = self._find_project_resources()

        if not resources:
            logging.info('No suitable resources found')
            return

        logging.info('Uploading data from %s resources to statface using %s threads', len(resources), MAX_PARALLEL)

        flow.parallel(self._upload_data_from_resource_to_statface, resources, MAX_PARALLEL)

    def _find_project_resources(self):
        attrs = {
            'type': 'json-reporter',
            'project': self.Parameters.project,
            'tool': 'hermione',
            'branch': 'dev'
        }

        if self.Parameters.service != '':
            attrs.update({'service': self.Parameters.service})

        platforms = self.Parameters.platforms

        if self.Parameters.project == 'web4':
            attrs.update({
                'report_description': 'merged'
            })

            platforms = None

        project_resources = []

        if platforms:
            for platform in platforms:
                attrs.update({'platform': platform})
                project_resources = project_resources.extend(self._find_resources(attrs))
        else:
            project_resources = self._find_resources(attrs)

        return project_resources

    def _find_resources(self, attributes):
        date_from = convert_to_date(self.Parameters.date_from)
        date_to = convert_to_date(self.Parameters.date_to)

        logging.info('-- date from: {}'.format(date_from))
        logging.info('-- date to: {}'.format(date_to))

        logging.info('-- project: {}'.format(attributes['project']))
        if 'service' in attributes:
            logging.info('-- service: {}'.format(attributes['service']))

        logging.info('-- tool: {}'.format(attributes['tool']))

        if 'platform' in attributes:
            logging.info('-- platform: {}'.format(attributes['platform']))

        res_params = {
            'resource_type': SANDBOX_CI_ARTIFACT,
            'attrs': attributes,
            'created': '{}..{}'.format(date_from, date_to),
            'state': ctr.State.READY
        }

        logging.info('Params of resources we are searching for:\n {}'.format(res_params))

        resources = load_all(sdk2.Resource.find(**res_params).order(-sdk2.Resource.id))
        resources = self._uniq_by_day(resources)

        logging.info('Found {} resources for given criteria'.format(len(resources)))

        return resources

    @staticmethod
    def _uniq_by_day(resources):
        resources_sorted = sorted(resources, key=attrgetter('created'), reverse=True)

        resources_filtered = []

        for key, group in groupby(resources_sorted, lambda res: res.created.strftime('%y-%m-%d')):
            resources_filtered.extend(islice(group, 1))

        return resources_filtered

    def _upload_data_from_resource_to_statface(self, resource):
        logging.info('Going to upload data from resource %s to statface', resource.id)

        resource_path = None

        try:
            resource_path = str(sdk2.ResourceData(resource).path)
        except ResourceNotAvailable as e:
            logging.info('Resource {} is not available'.format(resource.id))
            logging.info(e)

        if resource_path is None:
            return

        logging.info('Downloaded resource %s to %s', resource.id, resource_path)

        resource_file = open(resource_path)

        report_json = None

        try:
            report_json = json.loads(resource_file.read())
        except IOError as e:
            logging.info('Failed to parse json from %s', resource_path)
            logging.info(e)
        finally:
            resource_file.close()

        if report_json is None:
            return

        logging.info('Loaded json-report from resource file: %s', resource_path)

        gemini_auto_tests = {k: v for k, v in report_json.iteritems() if v['fullName'].find("Gemini autogenerated") != -1}

        logging.info('Filtered %s gemini-autogenerated tests from json-report', len(gemini_auto_tests.keys()))

        gemini_auto_tests = list(
            map(lambda (k, test): dict(
                fielddate=resource.created.strftime('%Y-%m-%d %H:%M:%S'),
                resource_id=resource.id,
                task_id=resource.task_id,
                browser_id=test['browserId'],
                name=test['fullName'],
                status=test['status'],
                file=test['file'],
                platform=test['meta']['platform'] if 'platform' in test['meta'] else '',
                app=test['meta']['app'] if 'app' in test['meta'] else '',
                testpalm_url=test['meta']['testpalmUrl'] if 'testpalmUrl' in test['meta'] else '',
                v_team=test['meta']['vTeam'] if 'vTeam' in test['meta'] else ''
            ), gemini_auto_tests.iteritems())
        )

        report_path = '{prefix}/{project}/{branch}/tests'.format(
            prefix=STATFACE_REPORT_PREFIX,
            project=self.Parameters.project,
            branch='dev'
        )

        report = self.statface.get_report(report_path)

        report.upload_data(
            scale='s',
            data=gemini_auto_tests,
            request_kwargs=dict(timeout=60),
            _request_timeout=60,
        )
