# -*- coding: utf-8 -*-

import itertools
import logging

from sandbox.common.types import task as ctt
from sandbox import sdk2

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.infrastats.base import InfratestInfrastatsBase
from sandbox.projects.sandbox_ci.infrastats.simple import InfratestInfrastats

TOOLS = ['gemini', 'hermione', 'hermione_e2e']

DEFAULT_IGNORE_TAGS = ['CASTLE', 'REPEATER']


class Project(object):
    WEB4 = 'web4'
    FIJI = 'fiji'
    NERPA = 'nerpa'
    TURBO = 'turbo'
    CHAT = 'chat'


class Platform(object):
    DESKTOP = 'desktop'
    TOUCH_PAD = 'touch-pad'
    TOUCH_PHONE = 'touch-phone'
    TV = 'tv'


class InfratestInfrastatsIntegral(InfratestInfrastatsBase):
    """
        Интеграционная задача для запуска INFRATEST_INFRASTATS для всех проектов,
        инструментов и платформ
    """

    class Parameters(InfratestInfrastatsBase.Parameters):
        description = 'Интеграционная задача для сбора статистической информации по плавающим тестам и ошибкам'

        with InfratestInfrastatsBase.Parameters.filter_block():
            use_web4 = sdk2.parameters.Bool(Project.WEB4, default=True)
            use_fiji = sdk2.parameters.Bool(Project.FIJI, default=True)
            use_nerpa = sdk2.parameters.Bool(Project.NERPA, default=True)
            use_turbo = sdk2.parameters.Bool(Project.TURBO, default=True)
            use_chat = sdk2.parameters.Bool(Project.CHAT, default=True)

        ignore_tags = sdk2.parameters.List('Ignore tags', sdk2.parameters.String, default=DEFAULT_IGNORE_TAGS)

        with sdk2.parameters.Group('Other') as other_block:
            _container = parameters.environment_container()

    def on_execute(self):
        with self.memoize_stage.mk_subtasks(commit_on_entrance=False):
            subtasks = []

            if self.Parameters.use_web4:
                subtasks.append(self.create_web4_subtasks())

            if self.Parameters.use_fiji:
                subtasks.append(self.create_fiji_subtasks())

            if self.Parameters.use_nerpa:
                subtasks.append(self.create_nerpa_subtasks())

            if self.Parameters.use_turbo:
                subtasks.append(self.create_turbo_subtasks())

            if self.Parameters.use_chat:
                subtasks.append(self.create_chat_subtasks())

            subtasks = list(itertools.chain.from_iterable(subtasks))

            raise sdk2.WaitTask(
                tasks=self.meta.start_subtasks(subtasks),
                statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )
        with self.memoize_stage.process():
            self.Context.report_resources = self.meta.get_subtasks_artifacts_ids()

    def create_web4_subtasks(self):
        return self.create_subtasks(
            project=Project.WEB4,
            tools=['hermione', 'hermione_e2e']
        )

    def create_fiji_subtasks(self):
        return self.create_subtasks(
            project=Project.FIJI,
            platforms=[Platform.DESKTOP, Platform.TOUCH_PAD, Platform.TOUCH_PHONE, Platform.TV],
            services=['images', 'video']
        )

    def create_nerpa_subtasks(self):
        return self.create_subtasks(
            project=Project.NERPA,
            platforms=[Platform.DESKTOP, Platform.TOUCH_PHONE],
            tools=['gemini', 'hermione']
        )

    def create_turbo_subtasks(self):
        return self.create_subtasks(
            project=Project.TURBO,
            platforms=[Platform.DESKTOP, Platform.TOUCH_PHONE]
        )

    def create_chat_subtasks(self):
        return self.create_subtasks(
            project=Project.CHAT,
            platforms=[Platform.DESKTOP, Platform.TOUCH_PAD, Platform.TOUCH_PHONE],
            tools=['hermione']
        )

    def create_subtasks(self, project, platforms=[None], tools=TOOLS, services=None):
        subtasks = []

        for tool in tools:
            for platform in platforms:
                if services:
                    for service in ['images', 'video']:
                        subtasks.append(self.create_subtask(project, tool, platform, service))
                else:
                    subtasks.append(self.create_subtask(project, tool, platform))

        return subtasks

    def create_subtask(self, project, tool, platform=None, service=None, **kwargs):
        logging.info(
            'create subtask for: project={} service={} tool={} platform={}'.format(project, service, tool, platform)
        )

        only_released_stable = project == 'web4'

        additional_tags = [platform] if platform else []

        return self.meta.create_subtask(
            task_type=InfratestInfrastats,
            _container=self.Parameters._container,
            project=project,
            service=service,
            tool=tool,
            platform=platform,
            date_from=self.Parameters.date_from,
            date_to=self.Parameters.date_to,
            branch=self.Parameters.branch,
            ignore_tags=self.Parameters.ignore_tags,
            only_released_stable=only_released_stable,
            description=self.get_subtask_description(project, service, tool, platform),
            additional_tags=additional_tags,
            notifications=[],
            **kwargs
        )

    def get_subtask_description(self, *args):
        return '-'.join(filter(lambda x: x, args))
