# -*- coding: utf-8 -*-

import os
from sandbox import sdk2
import logging
import shutil
import traceback
from sandbox.common.types import task as ctt
from sandbox.common.types import resource as ctr

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.resources import InfrastatsReport
from sandbox.projects.sandbox_ci.utils import flow
from sandbox.projects.sandbox_ci.utils.context import Debug, GitRetryWrapper, Node
from sandbox.projects.sandbox_ci.infrastats.base import InfratestInfrastatsBase
from sandbox.projects.sandbox_ci.infrastats.date_util import convert_to_date
from sandbox.agentr.errors import ResourceNotAvailable

ITEMS_PER_PAGE = 1000
RESOURCE_TYPE = 'json-reporter'
DUMPS_BASE_DIR = 'dumps'
REPORT_FOLDER = 'infrastats-report'
INFRASTATS_CUSTOM_TYPE = 'infrastats-custom'
REPORT_TTL = 365  # @see https://st.yandex-team.ru/FEI-9832
# Sandbox ограничивает количество одновременных скачивания в один раздел диска
# @see https://st.yandex-team.ru/SANDBOX-5712#1533906952000
# Делаем больше потоков, чем ограничение sandbox
MAX_PARALLEL = 20
TAGS_DELIMITER = ','


def get_type_by_project_and_tool(project, tool):
    if tool == 'hermione_e2e' and not project.upper() == 'FIJI':
        return 'SANDBOX_CI_{}'.format(tool.upper())

    return 'SANDBOX_CI_{}_{}'.format(project.upper(), tool.upper().replace('-', '_'))


def load_all(request):
    limited_req = request.limit(ITEMS_PER_PAGE)
    first_req = limited_req.offset(0)
    logging.debug('First request')
    res = list(first_req)
    logging.debug('Finished first request')

    logging.debug('Count')
    count = first_req.count
    logging.debug('Finished count')
    for offset in range(ITEMS_PER_PAGE, count, ITEMS_PER_PAGE):
        res.extend(limited_req.offset(offset))
    return res


class InfratestInfrastats(InfratestInfrastatsBase):
    """
        Таск для получения статистической информации по плавающим тестам и ошибкам,
        возникшим в ходе прогона gemini и hermione тестов
    """
    class Requirements(InfratestInfrastatsBase.Requirements):
        disk_space = 20 * 1024

    class Parameters(InfratestInfrastatsBase.Parameters):
        description = 'Cтатистическая информация по плавающим тестам и ошибкам'

        with InfratestInfrastatsBase.Parameters.filter_block():
            with sdk2.parameters.RadioGroup('Project') as project:
                project.values['web4'] = project.Value('web4', default=True)
                project.values['fiji'] = 'fiji'
                project.values['nerpa'] = 'nerpa'
                project.values['turbo'] = 'turbo'
                project.values['chat'] = 'chat'

            with sdk2.parameters.RadioGroup('Tool') as tool:
                tool.values['gemini'] = tool.Value('gemini', default=True)
                tool.values['hermione'] = 'hermione'
                tool.values['hermione_e2e'] = 'hermione_e2e'

            with sdk2.parameters.RadioGroup('Platform') as platform:
                platform.values[''] = platform.Value('---', default=True)
                platform.values['desktop'] = platform.Value('Desktop')
                platform.values['touch-pad'] = platform.Value('Touch-Pad')
                platform.values['touch-phone'] = platform.Value('Touch-Phone')
                platform.values['tv'] = platform.Value('TV')

            with sdk2.parameters.String('Service (fiji only)') as service:
                service.values[''] = service.Value('---', default=True)
                service.values['images'] = service.Value('Images')
                service.values['video'] = service.Value('Video')

            ignore_tags = sdk2.parameters.List('Ignore tags', sdk2.parameters.String)
            only_released_stable = sdk2.parameters.Bool('Only released stable',
                                                        description='use resources with `released:stable` attribute',
                                                        default=False)

        with sdk2.parameters.Group('Tasks') as tasks_block:
            tasks = sdk2.parameters.List('Tasks ids', sdk2.parameters.Integer)

        with sdk2.parameters.Group('Other') as other_block:
            _container = parameters.environment_container()
            wait_tasks = sdk2.parameters.List('Tasks to wait', sdk2.parameters.Integer)

    lifecycle_steps = {
        'npm_install': 'npm i @yandex-int/schetovod --production --registry=http://npm.yandex-team.ru',
        'generate_stats_report': 'node --max-old-space-size=8192 node_modules/.bin/schetovod -r {root_path} -d {dest_path}',
    }

    """
    Наличие свойста "project_name" и методов "project_dir" "working_path" "project_path"
    обусловлено желанием по максимуму использовать существующую функциональность
    из sandbox.projects.sandbox_ci.managers. Но код модулей managers предполагает использование
    совместно с sandbox.projects.sandbox_ci.task.BaseTask. И чтобы реиспользовать код пришлось
    добавить и переопределить вышеуказанные свойства и методы в данной задаче.
    """
    @property
    def project_name(self):
        return ''

    @property
    def project_dir(self):
        return self.working_path()

    def working_path(self, *args):
        return self.path(*args)

    def project_path(self, *args):
        return self.project_dir.joinpath(*args)

    @property
    def __infrastats_report_path(self):
        return self.project_path(REPORT_FOLDER)

    @property
    def report_description(self):
        return 'infrastats report'

    def on_prepare(self):
        self.lifecycle.update_vars(
            root_path=DUMPS_BASE_DIR,
            dest_path=self.__infrastats_report_path,
        )

        with GitRetryWrapper(), Node(self.Parameters.node_js_version):
            self.lifecycle('npm_install')

    def on_enqueue(self):
        super(InfratestInfrastats, self).on_enqueue()

        dependency_task_ids = self.Parameters.wait_tasks
        if not dependency_task_ids:
            logging.debug('No task dependencies to wait')
            return

        with self.memoize_stage.wait_tasks():
            logging.info('Will wait for task dependencies: {}'.format(dependency_task_ids))
            raise sdk2.WaitTask(dependency_task_ids, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def on_execute(self):
        logging.info('Infrastats start')

        resources = []

        tasks_ids = self.Parameters.tasks

        if tasks_ids:
            logging.info('Loading resources of specified tasks: {}'.format(tasks_ids))
            tasks = sdk2.Task.find(id=tasks_ids, children=True).limit(len(tasks_ids))
            resources = self._get_resources(tasks)
        else:
            resources = self._find_resources()

        if not resources:
            logging.info('No suitable resources found')
            return

        self._copy_resources(resources)

        os.environ['NODE_ENV'] = 'production'

        with GitRetryWrapper(), Node(self.Parameters.node_js_version), Debug('schetovod:*'):
            self.lifecycle('generate_stats_report')

        logging.info('create infrastats report')

        self._create_report(
            project=self.Parameters.project,
            branch=self.Parameters.branch,
            tool=self.Parameters.tool.replace('_e2e', '-e2e'),
            platform=self.Parameters.platform,
            service=self.Parameters.service,
            tasks=self.Parameters.tasks
        )

    def _find_resources(self):
        tool = self.Parameters.tool.replace('_e2e', '-e2e')

        attrs = {
            'type': 'json-reporter',
            'project': self.Parameters.project,
            'tool': tool,
            'platform': self.Parameters.platform
        }

        logging.info('-- project: {}'.format(self.Parameters.project))
        logging.info('-- tool: {}'.format(tool))
        logging.info('-- platform: {}'.format(self.Parameters.platform))

        if self.Parameters.branch != 'all':
            attrs.update({'branch': self.Parameters.branch})
            logging.info('-- branch: {}'.format(self.Parameters.branch))

        if self.Parameters.service != '':
            attrs.update({'project_service': self.Parameters.service})
            logging.info('-- service: {}'.format(self.Parameters.service))

        if self.Parameters.ignore_tags:
            logging.info('-- ignore tags: {}'.format(', '.join(self.Parameters.ignore_tags)))

        date_from = convert_to_date(self.Parameters.date_from)
        date_to = convert_to_date(self.Parameters.date_to)

        logging.info('-- date from: {}'.format(date_from))
        logging.info('-- date to: {}'.format(date_to))

        res_params = {
            'resource_type': SANDBOX_CI_ARTIFACT,
            'attrs': attrs,
            'created': '{}..{}'.format(date_from, date_to),
            'state': ctr.State.READY
        }

        logging.info('Params of resources we are searching for:\n {}'.format(res_params))

        resources = load_all(sdk2.Resource.find(**res_params).order(-sdk2.Resource.id))

        resources = self._filter_resources(resources)

        logging.info('Found {} resources for given criteria'.format(len(resources)))

        return resources

    def _filter_resources(self, resources):
        if not self.Parameters.ignore_tags:
            return resources

        def no_ignore_tags(resource):
            resource_tags = dict(resource).get('tags', '').split(TAGS_DELIMITER)
            for tag in self.Parameters.ignore_tags:
                if tag in resource_tags:
                    return False
            return True

        return filter(no_ignore_tags, resources)

    def _get_resources(self, tasks):
        res = []
        attrs = {'type': RESOURCE_TYPE}

        if self.Parameters.only_released_stable:
            attrs['released'] = 'stable'

        typed_res = load_all(sdk2.Resource.find(
            type=SANDBOX_CI_ARTIFACT,
            attrs=attrs,
            task=tasks,
            state=ctr.State.READY
        ))
        logging.info('Found %s resources of type %s', len(typed_res), RESOURCE_TYPE)
        res.extend(typed_res)
        return res

    def target_path(self, task_id):
        return self.path(DUMPS_BASE_DIR, str(task_id))

    @staticmethod
    def _uniq_by_task_id(items):
        task_ids = []
        res = []

        for i in items:
            if i.task_id not in task_ids:
                res.append(i)
                task_ids.append(i.task_id)

        return res

    def _copy_resources(self, resources):
        # https://st.yandex-team.ru/INFRADUTY-3638; удалить костыль после https://st.yandex-team.ru/FEI-12872
        resources = self._uniq_by_task_id(resources)

        logging.info('Copying %s resources using %s threads', len(resources), MAX_PARALLEL)

        # Папки для ресурсов создаем до многопоточных вызовов
        task_ids = [resource.task_id for resource in resources]
        for task_id in set(task_ids):
            self.target_path(task_id).mkdir(parents=True, exist_ok=True)

        flow.parallel(self._copy_resource, resources, MAX_PARALLEL)

    def _copy_resource(self, resource):
        logging.info('Copying resource %s', resource.id)

        # Для получения атрибута "type" воспользуемся интерфейсом итератора по атрибутам
        # Обращение resource.type возвращает python класс
        attrs = dict(resource)

        try:
            target_file = self.target_path(resource.task_id) / '{}.json'.format(attrs['type'])
            source_file = sdk2.ResourceData(resource).path
            logging.info('Copy resource %s to %s', source_file, target_file)
            shutil.copy(str(source_file), str(target_file))
        except ResourceNotAvailable as e:
            logging.info('Resource {} is not available'.format(resource.id))
            logging.info(e)
        except IOError:
            logging.error('error on copy file from %s to %s', source_file, target_file)
            logging.error(traceback.format_exc())

    def _create_report(self, tasks, **args):
        if tasks:
            type = INFRASTATS_CUSTOM_TYPE
        else:
            type = '-'.join(filter(lambda x: x, map(lambda f: args.get(f), ['project', 'tool', 'platform', 'service'])))

        self.artifacts.create_report(
            resource_type=InfrastatsReport,
            resource_path=self.__infrastats_report_path,
            type=type,
            status=ctt.Status.SUCCESS,
            add_to_context=True,
            public=True,
            root_path='index.html',
            ttl=REPORT_TTL,
            **args
        )
