# -*- coding: utf-8 -*-

import logging
import os


from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.sandboxsdk import process
import sandbox.projects.common.arcadia.sdk as arcadia_sdk


from sandbox.projects.sandbox_ci.infratest.apphost_service_create_package.utils import generate_content, find_review, parse_pr


class ApphostServiceCreatePackage(sdk2.Task):
    """
        Задача, создающая тип ресурса для хранения пакета шаблонов.
        Используется в npm-пакете @yandex-int/apphost-service.
    """

    class Parameters(sdk2.Parameters):
        project_resource = sdk2.parameters.String('Название типа ресурса', required=True)
        releasers = sdk2.parameters.List('Релизеры пакета', required=True)

        with sdk2.parameters.Output():
            arcanum_pr = sdk2.parameters.String('Номер Arcanum PR')
            user_message = sdk2.parameters.String('Сообщение для пользователя')

    def on_enqueue(self):
        same_tasks = list(sdk2.Task.find(
            type=ApphostServiceCreatePackage,
            status=(ctt.Status.Group.QUEUE + ctt.Status.Group.EXECUTE + ctt.Status.Group.WAIT),
            input_parameters=dict(
                project_resource=self.Parameters.project_resource,
                releasers=self.Parameters.releasers,
            ),
        ).order(-sdk2.Task.id).limit(0))

        same_tasks = filter(lambda task: task.id != self.id, same_tasks)

        if len(same_tasks) is not 0:
            raise sdk2.WaitTask(
                tasks=same_tasks,
                statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )

    def update_file(self, gen_path):
        new_content = None
        with open(gen_path, 'r') as gen_read:
            content = gen_read.read()

            new_content = generate_content(content, dict(name=self.Parameters.project_resource, releasers=self.Parameters.releasers))

            if new_content == content:
                return False

        with open(gen_path, 'w') as gen_write:
            gen_write.write(new_content)

        return True

    def on_execute(self):
        summary = 'sandbox_ci: add templates package resource {}'.format(self.Parameters.project_resource)
        commit_file_path = str(self.path('commit.txt'))
        with open(commit_file_path, 'w') as commit_file:
            commit = 'sandbox_ci: add templates package resource {}\n'\
                     'Автор: {}\n'\
                     'Создано задачей https://sandbox.yandex-team.ru/task/{}\n'\
                     .format(
                         self.Parameters.project_resource,
                         self.author,
                         self.id,
                     )
            commit_file.write(commit)

        gen_folder_path = 'sandbox/projects/sandbox_ci/resources'
        arcadia_url = sdk2.svn.Arcadia.trunk_url(gen_folder_path)
        arcadia_root = arcadia_sdk.do_clone(arcadia_url, self)
        sdk2.svn.Arcadia.update(os.path.join(arcadia_root, gen_folder_path), parents=True)

        gen_path = os.path.join(arcadia_root, gen_folder_path, 'generated.py')
        gen_updated = self.update_file(gen_path)

        if not gen_updated:
            self.Parameters.user_message = 'Тип ресурса не требует обновления'
            return

        pr = None
        try:
            pr = find_review(summary, sdk2.Vault.data('env.ARCANUM_OAUTH_TOKEN'))
        except Exception as e:
            logging.warn('error loading pr {}'.format(e.message))

        if not pr:
            cmd = ['svn', 'commit', '--username', 'robot-serp-bot', '--file', commit_file_path, '--encoding', 'UTF-8',
                   '--with-revprop', 'arcanum:check=yes',
                   gen_path]

            p = process.run_process(cmd, log_prefix='create_pr', work_dir=arcadia_root, check=False)
            with open(p.stdout_path, 'r') as out_file:
                out = out_file.read()
                pr = parse_pr(out)

            self.Parameters.user_message = 'Создан новый Arcanum PR'
        else:
            cmd = ['svn', 'commit', '--username', 'robot-serp-bot', '--file', commit_file_path, '--encoding', 'UTF-8',
                   '--with-revprop', 'arcanum:review-upload={}'.format(pr),
                   '--with-revprop', 'arcanum:review-publish=yes',
                   gen_path]

            p = process.run_process(cmd, log_prefix='update_pr', work_dir=arcadia_root, check=False)
            with open(p.stdout_path, 'r') as out_file:
                out = out_file.read()
                pr = parse_pr(out)

            self.Parameters.user_message = 'Обновлен существующий Arcanum PR'

        # svn commit фейлится в любом случае, поэтому
        # фейлим задачу, если не удалось вытащить из вывода команды коммита id PR
        if not pr:
            raise TaskFailure('В выводе команды svn нет номера PR, что-то пошло не так. '
                              'Напишите в https://wiki.yandex-team.ru/infraduty/')

        self.Parameters.arcanum_pr = pr
