# -*- coding: utf-8 -*-

import itertools
import logging

from sandbox import sdk2
from sandbox.common.types import task as ctt

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.task.ManagersTaskMixin import ManagersTaskMixin
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement
from sandbox.sandboxsdk.environments import PipEnvironment


def get_platform_cat(items):
    """
    Оптимистично вычисляем категорию для платформы
    """
    if 'yml_unknown' in items:
        return 'yml_unknown'

    if 'yml_yes_hermione_yes' in items:
        return 'yml_yes_hermione_yes'

    if 'yml_yes_hermione_no' in items:
        return 'yml_yes_hermione_no'

    if 'yml_no_hermione_yes' in items:
        return 'yml_no_hermione_yes'

    return 'yml_no_hermione_no'


def get_title_cat(items):
    """
    Строго вычисляем категорию для кейса
    """
    if 'yml_unknown' in items:
        return 'yml_unknown'

    if 'yml_yes_hermione_no' in items:
        return 'yml_yes_hermione_no'

    if 'yml_no_hermione_yes' in items:
        return 'yml_no_hermione_yes'

    if 'yml_yes_hermione_yes' in items:
        return 'yml_yes_hermione_yes'

    return 'yml_no_hermione_no'


# группировка
def by_title(x):
    return x['test']


def by_platform(x):
    return x['platform']


class InfratestAssertMetricsStats(TasksResourceRequirement, ManagersTaskMixin, sdk2.Task):
    """
        Таск сбора статистики покрытия assertMetrics
    """

    github_context = None

    class Requirements(sdk2.Requirements):
        environments = (
            PipEnvironment('python-statface-client', custom_parameters=["requests==2.18.4"]),
        )

    class Parameters(sdk2.Parameters):
        project = sdk2.parameters.String('Проект', default='web4')

        with sdk2.parameters.Group('Other') as other_block:
            _container = parameters.environment_container()
            wait_tasks = parameters.wait_tasks
            wait_output_parameters = parameters.wait_output_parameters

    def on_enqueue(self):
        wait_tasks = self.Parameters.wait_tasks
        if not wait_tasks:
            logging.debug('No task dependencies to wait')
            return

        with self.memoize_stage.wait_tasks():
            logging.info('Will wait for task dependencies: {}'.format(wait_tasks))
            raise sdk2.WaitTask(wait_tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def on_execute(self):
        full_report_path = 'Yandex.Productivity/infratest/{}/dev/assert-metrics'.format(
            self.Parameters.project
        )
        stats_report_path = 'Yandex.Productivity/infratest/{}/dev/assert-metrics-stats'.format(
            self.Parameters.project
        )

        full_report = self.statface.get_report(full_report_path)
        stats_report = self.statface.get_report(stats_report_path)

        date = full_report.fetch_dates(scale='d')[-1]

        data = full_report.download_data(scale=u'd', date_min=date, date_max=date)

        # категории и цифры за день
        categories = {}

        for title, title_tests in itertools.groupby(sorted(data, key=by_title), by_title):
            title_tests = list(title_tests)
            v_team = title_tests[0]['v_team']
            if v_team is None:
                print(title, title_tests)
                continue
            platform_cats = []

            for platform, platform_tests in itertools.groupby(sorted(title_tests, key=by_platform), by_platform):
                test_cats = list(map(lambda x: x['category'], list(platform_tests)))
                platform_cats.append(get_platform_cat(test_cats))

            title_cat = get_title_cat(platform_cats)

            if v_team not in categories:
                categories[v_team] = {
                    'yml_no_hermione_no': 0,
                    'yml_no_hermione_yes': 0,
                    'yml_unknown': 0,
                    'yml_yes_hermione_no': 0,
                    'yml_yes_hermione_yes': 0
                }

            v_team_cats = categories[v_team]

            if title_cat not in v_team_cats:
                v_team_cats[title_cat] = 0

            v_team_cats[title_cat] += 1

        new_data = []

        for (v_team, cats) in categories.items():
            new_data.append(dict(v_team=v_team, fielddate=date, **cats))

        stats_report.upload_data(scale=u'd', data=new_data)
