# -*- coding: utf-8 -*-

import logging
import time

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.sandbox_ci.task.ManagersTaskMixin import ManagersTaskMixin
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement

FINAL_STATUS = ctt.Status.Group.FINISH + ctt.Status.Group.BREAK


def datetime_to_unix_time(utc_dt):
    import pytz

    local_tz = pytz.timezone('Europe/Moscow')
    local_dt = utc_dt.replace(tzinfo=pytz.utc).astimezone(local_tz)
    normalized_dt = local_tz.normalize(local_dt)

    return int(time.mktime(normalized_dt.timetuple())) * 1000


class InfratestCastleStats(TasksResourceRequirement, ManagersTaskMixin, sdk2.Task):
    """
        Таск сбора статистики Castle из SB задач.
    """
    class Requirements(sdk2.Requirements):
        environments = (
            PipEnvironment('pytz'),
            PipEnvironment('python-statface-client', custom_parameters=["requests==2.18.4"]),
        )

    class Parameters(sdk2.Parameters):
        period_distance_hours = sdk2.parameters.Integer('Период в часах, за который будем обрабатывать записи в отчёте', default=48)
        statface_host = sdk2.parameters.String('Хост загрузки данных в statface', default='https://upload.stat.yandex-team.ru')
        report_id = sdk2.parameters.String('ID отчёта', default='Yandex/Infra/Castle/CastleUsageStatistics')

    def on_execute(self):
        self._cached_tasks = {}

        report = self.statface.get_old_report(self.Parameters.report_id)

        period_distance = self.Parameters.period_distance_hours * 60 * 60
        data = report.download_data(scale=u's', _period_distance=period_distance)
        logging.info('Download from statface %i record(s)', len(data))

        new_data = []
        for record in data:
            task_id = int(record['task_id'])
            # task_id - dimension (не может быть None), master_task_id - measure (может быть None)
            if not task_id or not record['master_task_id']:
                continue

            try:
                new_record = dict(record)
                need_update = False
                if not record['task_finish_time']:
                    task = self._get_finished_task(task_id)
                    if task:
                        new_record['task_finish_time'] = datetime_to_unix_time(task.updated)
                        new_record['task_finish_status'] = task.status
                        need_update = True

                if not record['master_task_start_time']:
                    master_task_id = int(record['master_task_id'])
                    master_task = self._get_finished_task(master_task_id)
                    if master_task:
                        new_record['master_task_start_time'] = datetime_to_unix_time(master_task.created)
                        new_record['master_task_finish_status'] = master_task.status
                        need_update = True

                if need_update:
                    new_data.append(new_record)
            except Exception as e:
                logging.error('Failed to append new record: %s', str(e))

        if new_data:
            report.upload_data(scale=u's', data=new_data)

    def _get_finished_task(self, task_id):
        if task_id not in self._cached_tasks:
            self._cached_tasks[task_id] = sdk2.Task.find(id=task_id, status=FINAL_STATUS).limit(1).first()
        return self._cached_tasks[task_id]
