# -*- coding: utf-8 -*-

import itertools


class BaseTaskManager(object):
    def __init__(self, branch, meta, artifacts):
        self.branch = branch
        self.meta = meta
        self.artifacts = artifacts
        self.description = 'gemini-testing/{}#{}'.format(self.tool_name, branch)

    def create_subtasks(self):
        subtasks = [
            self.create_web4_test_tasks(),
            self.create_fiji_test_tasks(),
            self.create_nerpa_test_tasks(),
            self.create_turbo_test_tasks(),
            self.create_redir_warning_test_tasks()
        ]

        return list(itertools.chain.from_iterable(subtasks))

    def create_web4_test_tasks(self):
        return self.create_test_subtasks(
            attrs_list=[{'type': 'web4'}, {'type': self.tool_name}, {'type': 'features'}],
            owner='serp',
            project='web4',
            platforms=['desktop', 'touch-pad', 'touch-phone'],
            selective_run=False
        )

    def create_nerpa_test_tasks(self):
        return self.create_test_subtasks(
            attrs_list=[{'type': 'nerpa'}, {'type': self.tool_name}],
            owner='news',
            project='nerpa',
            platforms=['desktop', 'touch-phone']
        )

    def create_turbo_test_tasks(self):
        return self.create_test_subtasks(
            attrs_list=[{'type': 'turbo'}, {'type': self.tool_name}],
            owner='serp',
            project='turbo',
            platforms=['touch-phone']
        )

    def create_fiji_test_tasks(self):
        subtasks = []

        for service in ['images', 'video']:
            subtasks += self.create_test_subtasks(
                attrs_list=[{'type': 'fiji'}],
                owner='mm-interfaces',
                project='fiji',
                platforms=['desktop', 'touch-pad', 'touch-phone'],
                description='{} {}'.format(self.description, service),
                service=service
            )

        return subtasks

    def create_redir_warning_test_tasks(self):
        return self.create_test_subtasks(
            attrs_list=[{'type': 'redir-warning'}, {'type': self.tool_name}],
            owner='serp',
            project='redir-warning',
            platforms=['desktop', 'touch-phone', 'touch-pad']
        )

    def create_test_subtasks(self, attrs_list, owner, project, platforms, description=None, **kwargs):
        resources = self.artifacts.get_dev_ready_resources(
            attrs_list=attrs_list,
            owner=owner,
            repo=project,
            YENV='testing',
        )

        subtasks = []

        for platform in platforms:
            subtasks.append(self.meta.create_subtask(
                task_type=self.tasks[project],
                build_artifacts_resources=resources,
                platform=platform,
                description='{} {}'.format(description or self.description, platform),
                project_build_context='dev',
                custom_command=self.get_custom_command(project),
                report_github_statuses=False,
                environ=self.environ,
                additional_tags=[platform],
                **kwargs
            ))

        return subtasks

    @property
    def environ(self):
        return {
            # Выключаем репортинг падений тестов
            'json_reporter_enabled': 'false',
            'stat_reporter_enabled': 'false'
        }

    def get_custom_command(self, project):
        cmds = []

        cmds.append('npm remove {}'.format(self.tool_name))
        cmds.append('npm i https://github.com/gemini-testing/{}/archive/{}.tar.gz'.format(self.tool_name, self.branch))

        return ' && '.join(cmds)
