# -*- coding: utf-8 -*-

import logging
import os
import tarfile

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import misc as ctm
from sandbox.sandboxsdk import process

from sandbox.projects.common.build.YaMake2 import YaMake2, YaMakeParameters
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement
from sandbox.projects.sandbox_ci.utils.context import Node


NPM_LOGIN_RESOURCE_ID = 1384098502


class NpmPublishParameters(YaMakeParameters):
    sandbox_container = parameters.environment_container()
    with sdk2.parameters.Group('Publish parameters') as publish_block:
        skip_publish = sdk2.parameters.Bool('Skip publishing', required=True, default=False)
        package_version = sdk2.parameters.String('Package version', required=True)
        npm_username = sdk2.parameters.String('NPM username', required=True)
        npm_password_vault_name = sdk2.parameters.String('NPM password vault name', required=True)
        npm_password_vault_owner = sdk2.parameters.String('NPM password vault owner', required=True)

    node_js = parameters.NodeJsParameters


class NpmPublish(TasksResourceRequirement, YaMake2):
    """
        Задача для публикации пакета из аркадии
    """
    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.LOCAL

    class Parameters(NpmPublishParameters):
        pass

    def post_build(self, source_dir, output_dir, pack_dir):
        if self.Parameters.skip_publish:
            logging.info('Package publishing skipped because of skip_publish')
            return

        if self.Parameters.arcadia_patch:
            logging.info('Package publishing skipped because of arcadia_patch')
            return

        horizon_dir = os.path.join(output_dir, 'serp', 'node_modules', 'horizon-protos')

        # права на файлы из клонированной аркадии недостаточны для изменения версии в package.json
        process.run_process(['chmod', '-R', 'a+w', horizon_dir], log_prefix='chmod_package', check=True)

        # распаковываем out.tar.gz, так как папка out почему-то не появляется после сборки
        with tarfile.open(os.path.join(horizon_dir, 'out.tar.gz'), 'r') as tar:
            tar.extractall(path=horizon_dir)

        process.run_process(['ls', '-alshR', horizon_dir], log_prefix='ls_hor_q')

        with Node(self.Parameters.node_js_version):
            version = self.Parameters.package_version
            process.run_process(['npm', 'version', version], work_dir=horizon_dir, log_prefix='npm_version', check=True)

            npm_login_resource = sdk2.Resource.find(id=NPM_LOGIN_RESOURCE_ID).first()
            logging.info('found npm script resource {}'.format(npm_login_resource))
            if not npm_login_resource:
                raise TaskFailure('Resource with npm-login script not found by id {}'.format(NPM_LOGIN_RESOURCE_ID))

            npm_login_data = sdk2.ResourceData(npm_login_resource)
            if not npm_login_data:
                raise TaskFailure('Unable to sync npm-login script from {}'.format(NPM_LOGIN_RESOURCE_ID))

            npm_login_path = str(npm_login_data.path.resolve())

            npm_username = self.Parameters.npm_username
            npm_password = sdk2.Vault.data(self.Parameters.npm_password_vault_owner, self.Parameters.npm_password_vault_name)
            npm_email = '{}@yandex-team.ru'.format(npm_username)

            cmd = [npm_login_path, '--no-pass-enc', npm_username, npm_password, npm_email]
            process.run_process(cmd, log_prefix='npm_login', check=False)
            process.run_process(['npm', 'whoami', '--registry', 'http://npm.yandex-team.ru/'], work_dir=horizon_dir, log_prefix='npm_whoami', check=True)
            process.run_process(['npm', 'publish', '--registry', 'http://npm.yandex-team.ru/'], work_dir=horizon_dir, log_prefix='npm_publish', check=True)
