# -*- coding: utf-8 -*-

import logging
import traceback

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.common.errors import TaskFailure

from sandbox.projects.sandbox_ci.utils import flow

from .base_builder import BuildDone
from .recipe import Recipe
from .utils import flatten

PENDING_TASKS_KEY = 'pending_task_ids'


class Scheduler(object):
    @staticmethod
    def init(recipes, default_version=None):
        state = {
            'recipes': [],
            'has_errors': False,
            'summary': [],
        }

        for index, recipe_config in enumerate(recipes):
            if recipe_config.get('enabled') is False:
                continue

            recipe_config['package'] = dict(
                {'version': default_version},
                **(recipe_config.get('package') or {})
            )
            recipe_state = Recipe.init(recipe_config, index)
            state['recipes'].append(recipe_state)

        return state

    def __init__(self, task, publisher, state):
        self._task = task
        self._publisher = publisher
        self._state = state
        self._recipes = map(lambda recipe_state: Recipe(task, recipe_state), state['recipes'])

    def process(self):
        while len(self._recipes):
            build_tasks = flatten(flow.parallel(self._build_package, self._recipes))
            if len(build_tasks):
                raise sdk2.WaitTask(build_tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=False)

        if self._state['has_errors']:
            raise TaskFailure('Some recipes failed.')

    def _build_package(self, recipe):
        try:
            task_groups = self._group_tasks(recipe.get(PENDING_TASKS_KEY, []))
            if task_groups['broken']:
                self._stop_tasks(task_groups['pending'])
                raise Exception('Package "{}" build subtasks ({}) are broken.'
                                .format(recipe.label, task_groups['broken']))

            if not task_groups['pending']:
                logging.info('Building package "{}"...'.format(recipe.label))
                builder = recipe.get_builder()
                pending_tasks = [builder.build()]
                recipe.set(PENDING_TASKS_KEY, map(int, flatten(pending_tasks)))
            else:
                recipe.set(PENDING_TASKS_KEY, map(int, task_groups['pending']))

            logging.info('"{}" is waiting for {}.'.format(recipe.label, recipe.get(PENDING_TASKS_KEY, [])))
            return recipe.get(PENDING_TASKS_KEY, [])
        except BuildDone as done:
            self._publish_package(recipe, done.results)
        except Exception:
            self._handle_exception(recipe)

        return []

    def _publish_package(self, recipe, results):
        try:
            composer = recipe.get_composer()
            path = composer.compose(*results)
            self._publisher.publish(path)
            self._handle_success(recipe)
        except Exception:
            self._handle_exception(recipe)

    def _handle_success(self, recipe):
        success_message = '"{}" has been successfully published.'.format(recipe.label)
        logging.info(success_message)
        self._task.set_info(success_message)
        self._state['summary'].append([True, recipe.label])
        self._handle_any_outcome(recipe)

    def _handle_exception(self, recipe):
        error_message = traceback.format_exc()
        logging.error(error_message)
        self._task.set_info('Recipe for "{}" failed:\n{}'.format(recipe.label, error_message))
        self._state['has_errors'] = True
        self._state['summary'].append([False, recipe.label])
        self._handle_any_outcome(recipe)

    def _handle_any_outcome(self, recipe):
        # forget about the recipe in case of any exception, including BuildDone
        self._recipes.remove(recipe)
        self._state['recipes'].remove(recipe.raw)
        logging.info('No more actions on "{}".'.format(recipe.label))

    def _stop_tasks(self, tasks):
        map(lambda task: task.stop(), tasks)

    def _group_tasks(self, task_ids):
        finished = []
        broken = []
        pending = []
        for task_id in task_ids:
            task = sdk2.Task[task_id]
            if task.status in ctt.Status.Group.FINISH:
                finished.append(task)
            elif task.status in ctt.Status.Group.BREAK:
                broken.append(task)
            else:
                pending.append(task)

        return {'finished': finished, 'broken': broken, 'pending': pending}
