# -*- coding: utf-8 -*-

import os

from sandbox import sdk2
from sandbox.common.utils import singleton_property
from sandbox.common.types import misc as ctm

from sandbox.projects.sandbox_ci import parameters, managers
from sandbox.projects.sandbox_ci.utils import env
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper, Node
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement


class InfratestSkipCollector(TasksResourceRequirement, sdk2.Task):
    """
        Таск для сбора информации о заскипанных gemini/hermione тестах
    """
    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = 1024  # 1 Gb

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.String('Проект', required=True) as project:
            project.values['web4'] = 'web4'
            project.values['fiji'] = 'fiji'
            project.values['nerpa'] = 'nerpa'
            project.values['butterfly'] = 'butterfly'
            project.values['health'] = 'health'
            project.values['profile'] = 'profile'
            project.values['tutor'] = 'tutor'
            project.values['ugc-pages'] = 'ugc-pages'
            project.values['ydo'] = 'ydo'
            project.values['ydo-admin'] = 'ydo-admin'

        publish_skips_list = sdk2.parameters.Bool('Опубликовать плоский список заскипов')
        publish_grouped_skips = sdk2.parameters.Bool('Опубликовать список заскипов, сгруппировнный в таблицу')
        package_name = sdk2.parameters.String('Имя пакета skip-collector для npm install',
                                              default='@yandex-int/skip-collector')

        with sdk2.parameters.Group('Advanced'):
            _container = parameters.environment_container()

        node_js = parameters.NodeJsParameters

    lifecycle_steps = {
        'npm_install': 'npm i {package_name} --registry=http://npm.yandex-team.ru',
        'run': './node_modules/.bin/skip-collector --project {project} {publish_skips_list} {publish_grouped_skips}'
    }

    @singleton_property
    def lifecycle(self):
        return managers.LifecycleManager(
            task=self,
            steps=self.lifecycle_steps,
            variables=dict(
                package_name=self.Parameters.package_name,
                project=self.Parameters.project,
                publish_skips_list='--publish-skips-list' if self.Parameters.publish_skips_list else '',
                publish_grouped_skips='--publish-grouped-skips' if self.Parameters.publish_grouped_skips else '',
            )
        )

    @singleton_property
    def vault(self):
        return managers.VaultManager(self)

    @singleton_property
    def config(self):
        return managers.ConfigManager(self)

    def on_prepare(self):
        all_conf = self.config.get_properties()
        envs = (
            self.config.get_project_conf(all_conf, {'project_name': '*'}).get('environ', {}),
            env.from_vault(self),
        )
        env.export(env.merge(envs))
        os.environ['SYNCHROPHAZOTRON_PATH'] = str(self.synchrophazotron)
        env.log(self)

    def on_execute(self):
        with GitRetryWrapper(), Node(self.Parameters.node_js_version), self.vault.ssh_key():
            self.lifecycle('npm_install')
            self.lifecycle('run')
