# -*- coding: utf-8 -*-

import datetime as dt
import time
from sandbox import sdk2
from itertools import groupby

from sandbox.common.types import misc as ctm
from sandbox.projects.common.yasm import push_api
from sandbox.projects.sandbox_ci.utils import flow
from sandbox.projects.sandbox_ci.managers.profiler import get_history_tracker


STATFACE_REPORT = 'Yandex.Productivity/actions-profile'


def flatten(x):
    if isinstance(x, list):
        return [a for i in x for a in flatten(i)]
    else:
        return [x]


def utc_to_moscow(sb_time, to_string=False):
    format = '%Y-%m-%d %H:%M:%S'
    datetime = dt.datetime.strptime(sb_time.split('.')[0].replace('T', ' '), format) + dt.timedelta(hours=3)
    return datetime.strftime(format) if to_string else datetime


def time_to_int(sb_time):
    try:
        date = utc_to_moscow(sb_time)

        return int(time.mktime(date.timetuple()))
    except Exception:
        return 0


def get_tasks_by_ids(task_ids):
    return [sdk2.Task.find(id=task_id, children=True).limit(1).first() for task_id in task_ids]


def get_wait_tasks(task):
    subtasks_ids = task.Context.subtasks
    subtasks = [] if subtasks_ids is ctm.NotExists else get_tasks_by_ids(subtasks_ids)
    wait_subtasks = [subtask for subtask in subtasks if 'DONT_WAIT' not in subtask.Parameters.tags]
    wait_tasks = [t.id for t in wait_subtasks] + task.Parameters.wait_tasks

    return ','.join(str(t) for t in wait_tasks)


class ActionsStatSender(object):
    def __init__(self, task):
        self.__statface = task.statface.get_report(STATFACE_REPORT)
        self._task = task

    def execute(self):
        report_data = self.format_report_data(self._task)

        self.__statface.upload_data(
            scale='s',
            data=report_data,
            request_kwargs=dict(timeout=60),
            _request_timeout=60
        )

        flow.parallel(self.__upload_data_to_yasm, self.__format_yasm_data(report_data))

    def __upload_data_to_yasm(self, data):
        push_api.push_signals(
            signals={
                '{}execution_time_hgram'.format(action['signal_prefix']): [action['duration'], 1]
                for action in data['actions']
            },
            tags={
                'itype': 'sandboxci',
                'service_name': data['service_name'],
                'task_type': data['task_type'],
                'build_context': data['build_context'] or 'default'
            }
        )

    def __format_yasm_data(self, data):
        def grouper(i):
            return {'service_name': i['service_name'], 'task_type': i['task_type'], 'build_context': i['build_context']}

        formatted = []
        for i in data:
            formatted.append({
                'service_name': i['project'],
                'build_context': i['build_context'],
                'duration': i['time_stopped'] - i['time_started'],
                'task_type': i['action_name'] if i['task_id'] else i['parent_name'],
                'signal_prefix': '' if i['task_id'] else i['action_name'] + '_',
            })

        grouped = []
        for key, group_items in groupby(formatted, key=grouper):
            actions = [{'signal_prefix': i['signal_prefix'], 'duration': i['duration']} for i in group_items]
            key.update({'actions': actions})
            grouped.append(key)

        return grouped

    def format_report_data(self, task, report_data=[]):
        data = [self.format_task_data(task), self.format_action_data(task)]

        subtask_ids = task.Context.subtasks
        if subtask_ids is not ctm.NotExists:
            subtasks = get_tasks_by_ids(subtask_ids)

            return flatten(data + [self.format_report_data(st, report_data) for st in subtasks])

        return report_data + data

    def format_task_data(self, task):
        history_tracker = get_history_tracker(task)
        start_time = history_tracker.get_task_start_time()

        return {
            'fielddate': utc_to_moscow(start_time, to_string=True),
            'project': task.project_name,
            'parent_id': task.parent.id if task.parent else 0,
            'parent_name': task.parent.type.name if task.parent else '-',
            'task_id': task.id,
            'action_name': task.type.name,
            'wait_tasks': get_wait_tasks(task),
            'time_started': time_to_int(start_time),
            'time_stopped': time_to_int(history_tracker.get_task_finish_time()),
            'tags': ','.join(task.Parameters.tags),
            'build_context': task.Parameters.project_build_context if hasattr(task.Parameters, 'project_build_context') else '',
        }

    def format_action_data(self, task):
        task_steps = task.Context.task_steps

        if task_steps is not ctm.NotExists:
            return [{
                'fielddate': utc_to_moscow(step['start_utc_datetime'], to_string=True),
                'project': task.project_name,
                'parent_id': task.id,
                'parent_name': task.type.name,
                'task_id': 0,
                'action_name': step['tag'],
                'wait_tasks': '',
                'time_started': time_to_int(step['start_utc_datetime']),
                'time_stopped': time_to_int(step['finish_utc_datetime']),
                'tags': ','.join(task.Parameters.tags),
                'build_context': task.Parameters.project_build_context if hasattr(task.Parameters, 'project_build_context') else '',
            } for ind, step in enumerate(task_steps)]

        return []
