# -*- coding: utf-8 -*-

import os
import json
import logging

from sandbox.projects.sandbox_ci.utils.process import run_process
from sandbox.projects.sandbox_ci.utils.list_utils import flatten
from sandbox.projects.sandbox_ci.utils.startrek import parse_issue_keys

ARC_BIN = 'arc'


def arc_checkout(mount_point, commit):
    logging.debug('Arc checkout in {} to {} commit'.format(mount_point, commit))
    run_process(
        [ARC_BIN, 'checkout', commit],
        work_dir=mount_point,
        log_prefix='arc_checkout'
    )


def arc_push(mount_point, commit):
    logging.debug('Arc commit {} in {}'.format(commit, mount_point))
    run_process(
        [ARC_BIN, 'push', commit],
        work_dir=mount_point,
        log_prefix='arc_push'
    )


def arc_changed_files_cmd():
    return 'DIFF_OUT=$(arc log -n 2 --oneline ./ | cut -d " " -f 1) && arc diff --name-status $(echo $DIFF_OUT | cut -d " " -f 2) $(echo $DIFF_OUT | cut -d " " -f 1) ./ | cut -d "/" -f4-'


def arc_deleted_files_cmd():
    return 'DIFF_OUT=$(arc log -n 2 --oneline ./ | cut -d " " -f 1) && arc diff --name-status $(echo $DIFF_OUT | cut -d " " -f 2) $(echo $DIFF_OUT | cut -d " " -f 1) ./ | grep ^D | cut -d "/" -f4-'


def save_arc_changed_files(work_dir, file_path):
    logging.debug('Saving changed files for {} to {}'.format(work_dir, file_path))
    run_process(
        [arc_changed_files_cmd(), '>', str(file_path)],
        work_dir=work_dir,
        log_prefix='save_arc_changed_files'
    )


def save_arc_changed_files_with_statuses(work_dir, file_path):
    logging.debug('Saving changed files with statuses for last commit in {} to {}'.format(work_dir, file_path))
    p = run_process(
        ['arc log -n 1 --name-status --json .'],
        work_dir=work_dir,
        log_prefix='save_arc_changed_files_with_statuses',
    )

    with open(p.stdout_path, 'r') as f:
        data = json.load(f)

    with open(os.path.join(str(work_dir), file_path), 'w') as f:
        project_dir = data[0]['path']
        files = map(lambda f: {'status': f['status'], 'path': os.path.relpath(f['path'], project_dir)}, data[0]['names'])
        json.dump(files, f)


def get_arc_last_merge_commit_issues(work_dir):
    logging.debug('Getting last merge commit issues for {}'.format(work_dir))
    p = run_process(
        ['arc log -n 1 --json .'],
        work_dir=work_dir,
        log_prefix='arc_log_merge_commit',
    )

    with open(p.stdout_path, 'r') as f:
        return parse_issue_keys(json.load(f)[0]['message'].split())


def prefetch_files(workdir, name=['*.*'], include=['.'], exclude=[], **kwargs):
    name = flatten(name)
    include = flatten(include)
    exclude = flatten(exclude)

    logging.debug('Prefetching {} from {} in {} excluding {}'.format(', '.join(name), ', '.join(include), workdir, ', '.join(exclude)))

    find_name_opts = ' -o '.join(map(lambda i: '-name \'{}\' -print'.format(i), name))
    find_exclude_opts = ' '.join(map(lambda i: '-path \'{}\' -prune -o'.format(i), exclude))
    find_cmd = ['find', ' '.join(include), find_exclude_opts, find_name_opts]

    return run_process(
        find_cmd + ['|'] + ['xargs', 'head', '-c1'] + ['&&'] + ['echo \'Done\''],
        work_dir=workdir,
        log_prefix='arc_prefetch_files',
        **kwargs
    )
