# -*- coding: utf-8 -*-

import os
import logging


from sandbox.common.utils import get_task_link
from sandbox.projects.sandbox_ci.utils.arcanum import ArcanumApi, convert_sandbox_status_to_arcanum_status


class ArcanumStatusesManager(object):
    def __init__(self, task):
        """
        :param task:
        :type task: sandbox.projects.sandbox_ci.task.BaseTask
        """
        arcanum_api_token = os.environ.get('ARCANUM_API_OAUTH_TOKEN')
        if arcanum_api_token:
            self._api = ArcanumApi(token=arcanum_api_token)
        else:
            # В on_enqueue нет доступка к секретам, поэтому будем делать запросы через специальный сервер, который их
            # добавит.
            # @see: SANDBOX-3882, FEI-10158, FEI-16547
            self._api = ArcanumApi(host='github-token-proxy.si.yandex-team.ru')

        self.task = task

    @property
    def report_statuses_config(self):
        """
        :rtype: bool
        """
        return self.task.project_conf.get('arcanum', {}).get('report_statuses', False)

    @property
    def should_report_statuses(self):
        """
        По параметрам таски и конфигу вычисляет, должна ли она репортить статус.
        :rtype: bool
        """
        return self.task.Parameters.report_arcanum_checks and self.report_statuses_config

    def report_self_status(self, state=None, description=''):
        self.create_arcanum_check(
            self.task.Parameters.arcanum_review_request_id,
            self.task.Parameters.arcanum_diff_set_id,
            self.task.github_context,
            state if state else convert_sandbox_status_to_arcanum_status(self.task.status),
            get_task_link(self.task.id),
            description
        )

    def safe_report_self_status(self, state=None, description=''):
        try:
            self.report_self_status(state, description)
        except Exception as e:
            logging.debug('Unable to create Arcanum {state} check ({description}): {error}'.format(
                state=state,
                description=description,
                error=e.message,
            ))

    # На данный момент мы не оперируем sha коммитов,
    # мы ориентируемся на review_request_id и diff_set_id, имя для совместимости
    # читать как "репортим для текущего состояния кода"
    def report_status_to_current_sha(self, context, state, url, description, force=False):
        self.safe_report_status(
            self.task.Parameters.arcanum_review_request_id,
            self.task.Parameters.arcanum_diff_set_id,
            context,
            state,
            url,
            description,
            force
        )

    def safe_report_status(self, review_request_id, diff_set_id, context, state, url, description='', force=False):
        """
        Безопасно репортит чек дифф-сета в ревью-реквест с указанными параметрами, если этого требует задача и репортинг
        включён в genisys
        :param review_request_id: номер ревью-реквеста
        :type review_request_id: int
        :param diff_set_id: id дифф-сета
        :type diff_set_id: int
        :param check_type: контекст github статуса
        :type check_type: str
        :param status: статус дифф сета
        :type status: str
        :param url: адрес, на который ведёт чек
        :type url: str
        :param description: описание чека
        :type description: str
        :param force: отправить чек независимо от параметров задачи и конфигурации в genisys
        :type force: bool
        :rtype: None
        """
        try:
            self.create_arcanum_check(
                review_request_id=review_request_id,
                diff_set_id=diff_set_id,
                check_type=context,
                status=state,
                url=url,
                description=description,
                force=force,
            )
        except Exception as e:
            logging.debug('Unable to create Arcanum {state} check {context} (${description}): {error}'.format(
                state=state,
                description=description,
                context=context,
                error=e.message,
            ))

    def create_arcanum_check(self, review_request_id, diff_set_id, check_type, status, url='', description='', force=False):
        """
        Репортит чек дифф-сета в ревью-реквест с указанными параметрами
        :param review_request_id: номер ревью-реквеста
        :type review_request_id: int
        :param diff_set_id: id дифф-сета
        :type diff_set_id: int
        :param check_type: контекст github статуса
        :type check_type: str
        :param status: статус дифф сета
        :type status: str
        :param url: адрес, на который ведёт чек
        :type url: str
        :param description: описание чека
        :type description: str
        :param force: отправить чек независимо от параметров задачи и конфигурации в genisys
        :type force: bool
        :rtype: None
        """
        if not force and not self.should_report_statuses:
            logging.debug('Skip creating {status} Arcanum check [{type}] ({description})'.format(
                status=status,
                type=check_type,
                description=description
            ))
            return

        logging.debug('Create {status} Arcanum check [{type}] ({description})'.format(
            status=status,
            type=check_type,
            description=description
        ))

        self._api.create_check(
            review_request_id,
            diff_set_id,
            check_type,
            status,
            url,
            description
        )
