# -*- coding: utf-8 -*-

import os
import logging
import platform
import shutil
import pathlib2
import subprocess
import errno
import tempfile

from sandbox import sdk2
from sandbox.common.utils import singleton_property, Enum
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.common.platform import get_arch_from_platform

from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.projects.sandbox_ci.utils.process import run_process
from sandbox.projects.sandbox_ci.utils.github import GitHubApi
from sandbox.projects.sandbox_ci.utils import flow
from sandbox.projects.resource_types import REPORT_TEMPLATES_PACKAGE, REPORT_STATIC_PACKAGE
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.resources import InfrastatsReport

# Sandbox ограничивает количество одновременных скачиваний в один раздел диска
# @see https://st.yandex-team.ru/SANDBOX-5712#1533906952000
# Делаем больше потоков, чем ограничение sandbox
MAX_PARALLEL = 20

TAR_GZ_EXT = '.tar.gz'
SQUASHFS_EXT = '.squashfs'


class ArtifactCacheStatus(Enum):
    """ Artifact cache statuses. """

    FOUND = None
    IGNORED = None
    NOT_FOUND = None


class ArtifactsManager(object):
    """
    Manager that helps to work with artifact resources.
    """

    def __init__(self, task):
        """
        :param task: Task instance.
        """
        self.task = task

    @property
    def use_overlayfs(self):
        return getattr(self.task, 'use_overlayfs', False)

    @singleton_property
    def _github(self):
        return GitHubApi()

    def create_project_artifact_resource(self, relative_path, resource_type, **kwargs):
        """
        Creates resource with `project` attribute.
        :param relative_path: Path to resource file or directory relative to the task root directory.
        :param resource_type: Resource type class.
        :param kwargs: Additional attributes for resource to create.
        :return: Resource descriptor instance.
        """
        attrs = dict({'project': self.task.project_name}, **kwargs)
        return self.create_artifact_resource(relative_path, resource_type, **attrs)

    def create_reports(self, reports, parallel=False):
        if parallel:
            flow.parallel(self.create_report_unpacked, reports)
        else:
            [self.create_report(**r) for r in reports]

    def create_report_unpacked(self, attrs):
        return self.create_report(**attrs)

    def create_report(self, resource_type=SANDBOX_CI_ARTIFACT, resource_path=None, add_to_context=False, make_tar=False, **kwargs):
        """
        :param resource_path: path to resource relative to project or absolute path to resource
        :param add_to_context: appends generated resource id to `self.task.Context.report_resources`
        :param make_tar: makes tar report
        :param kwargs: additional attributes of created resource
        :return: Resource descriptor instance
        """
        # if resource_path is absolute, working_abs_path will become resource_path
        working_abs_path = self.task.working_path(self.task.project_name) / resource_path

        if not working_abs_path.exists():
            logging.debug('Not adding resource {}: no such path'.format(working_abs_path))
            return

        # sandbox fails to create resource with empty directory
        if working_abs_path.is_dir() and not any(working_abs_path.iterdir()):
            logging.debug('Not adding resource {}: directory is empty'.format(working_abs_path))
            return

        if working_abs_path.is_dir() and make_tar:
            working_abs_path = self.__make_tar(working_abs_path)

        abs_path = self.__sync_artifact(working_abs_path)

        """
        По умолчанию сохраняем ресурс в статусе SUCCESS
        и храним в течение месяца (30 дней)
        """
        attributes = dict(
            {
                'status': ctt.Status.SUCCESS,
                'ttl': 30,
            },
            **kwargs
        )

        if not attributes.get('report_description') and hasattr(self.task, 'report_description'):
            attributes['report_description'] = self.task.report_description

        if hasattr(self.task.Parameters, 'project_tree_hash'):
            attributes['project_tree_hash'] = self.task.Parameters.project_tree_hash

        if hasattr(self.task.Parameters, 'ref'):
            attributes['branch'] = self.task.Parameters.ref

        rel_path = abs_path.relative_to(self.task.path())
        resource = self.create_project_artifact_resource(
            resource_type=resource_type,
            relative_path=str(rel_path),
            **attributes
        )

        if add_to_context:
            self.task.Context.report_resources.append(resource.id)

        sdk2.ResourceData(resource).ready()

        return resource

    def duplicate_artifact_from_task_log(self, artifact_path, destination_dir_name='temp_task_log'):
        """
        :param artifact_path: path to resource relative to logs dir (TASK_LOGS resource)
        :return: copied artifact path
        """
        # нельзя один файл поместить в два ресурса, поэтому скопируем во временную папку
        source = self.task.log_path(artifact_path)
        destination = self.task.working_path(destination_dir_name) / artifact_path

        if source.exists():
            self.__copy_artifact(source, destination)
        else:
            logging.debug('Not duplicating artifact {}: no such path'.format(source))

        # проверка существования будет в create_report
        return destination

    def __make_tar(self, working_abs_path):
        archive_path = '{}.tar.gz'.format(working_abs_path)

        subprocess.call([
            'tar',
            '--directory=' + str(working_abs_path),
            '-zcf',
            archive_path,
            '.'
        ])

        return pathlib2.Path(archive_path)

    def __sync_artifact(self, working_abs_path):
        abs_path = self.get_artifact_path(working_abs_path)

        if working_abs_path != abs_path:
            self.__copy_artifact(working_abs_path, abs_path)

        return abs_path

    def __copy_artifact(self, src, dst):
        dir_path = os.path.dirname(str(dst))
        # не используем pathlib2 + exist_ok, так как при параллельном создании одной и той же папки в нескольких процесах все равно иногда стреляет "EEXIST"
        try:
            os.makedirs(dir_path)
        except OSError as e:
            if e.errno != errno.EEXIST:
                raise

        if src.is_dir():
            shutil.copytree(str(src), str(dst))
        else:
            shutil.copy2(str(src), str(dst))

    def get_artifact_path(self, working_abs_path):
        # We need to move artifact if task is running on ramdrive
        working_rel_path = working_abs_path.relative_to(self.task.working_path())

        return self.task.path() / working_rel_path

    def create_artifact_resource(self, relative_path, resource_type, **kwargs):
        """
        Create resource. `checksum` attribute will be treated special way and go to the resource description.
        :param relative_path: Path to resource file or directory relative to the task root directory.
        :param resource_type: Resource type class.
        :param kwargs: Additional attributes for resource to create.
        :return: Resource descriptor instance.
        """
        checksum = kwargs.get('checksum')
        default_description = '{}#{}'.format(relative_path, checksum) if checksum else relative_path
        description = kwargs.get('description', default_description)

        kwargs.update(path=str(relative_path))
        kwargs.update(description=description)

        return resource_type(task=self.task, **kwargs)

    def create_ready_artifact_resource(self, relative_path, resource_type, **kwargs):
        res = self.create_artifact_resource(relative_path, resource_type, **kwargs)

        sdk2.ResourceData(res).ready()

        return res

    def get_last_project_artifact_resource(self, resource_type, **kwargs):
        """
        Returns latest resource of the specified resource type with `project` attribute.
        :param resource_type: Resource type class.
        :param kwargs: Additional attributes to search resource for.
        :return: Resource descriptor instance.
        """
        attrs = dict(project=self.task.project_name, **kwargs)
        return self.get_last_artifact_resource(resource_type, **attrs)

    def get_last_artifact_resource(self, resource_type, state=ctr.State.READY, task_id=None, created=None, **kwargs):
        """
        Returns latest resource of the specified resource type.
        :param resource_type: Resource type class.
        :param kwargs: Additional attributes to search resource for.
        :return: Resource descriptor instance.
        """
        logging.debug('Looking for resource {} with attrs {}'.format(resource_type, kwargs))

        current_arch = get_arch_from_platform(platform.platform())

        params = dict(attrs=kwargs, state=state, arch=current_arch)
        if task_id:
            params['task_id'] = task_id

        if created:
            params['created'] = created

        return next(iter(resource_type.find(**params).limit(1)), None)

    def get_artifact_resource_by_id(self, resource_id):
        """
        Returns resource by the specified resource id.
        :param resource_id: Resource id.
        :return: Resource descriptor instance.
        """
        current_arch = get_arch_from_platform(platform.platform())
        return next(iter(sdk2.Resource.find(id=resource_id, arch=current_arch).limit(1)), None)

    @staticmethod
    def sync_build_artifact(resource):
        """
        Syncs specified artifact resource.
        :param resource: Resource descriptor instance.
        :return: Path to resource.
        """
        logging.info('Syncing build artifact resource: {}'.format(str(resource)))
        return sdk2.ResourceData(resource).path

    @staticmethod
    def sync_build_artifacts(resources):
        """
        Syncs specified artifact resources.
        :param resources: Iterable of resource descriptor instances.
        :return: List of paths to resources.
        """
        logging.info('Syncing build artifacts resources: {}'.format(', '.join([str(res_id) for res_id in resources])))
        return [sdk2.ResourceData(res_id).path for res_id in resources]

    @staticmethod
    def mount_build_artifact(resource):
        """
        Syncs and mounts specified artifact resources.
        :param resources: Iterable of resource descriptor instances.
        :return: List of paths to mounted resources.
        """
        logging.info('Mounting build artifact resource: {}'.format(str(resource)))
        return sdk2.ResourceData(resource).mount()

    @in_case_of('use_overlayfs', 'unpack_build_artifacts_in_overlayfs_mode')
    def unpack_build_artifacts(self, resources, target_path, strip_components=None, unsquash=False):
        """
        Syncs and then unpacks or mounts specified artifact resources to the target directory.
        :param resources: Iterable of resource descriptor instances.
        :param target_path: Path to the target directory (will be created if not exist).
        :param strip_components: Number of path components to strip during unpacking.
        :param unsquash: Unsquash image to target_path.
        """
        def get_artifact_ext(resource):
            attrs = dict(sdk2.Resource[resource])
            type = attrs['type'] if 'type' in attrs else ''
            return os.path.splitext(type)[1]

        def is_tar_artifact(resource):
            return not is_sqsh_artifact(resource)

        def is_sqsh_artifact(resource):
            return get_artifact_ext(resource) == '.sqsh'

        def sync_tar_build_artifacts(artifacts):
            artifacts_paths = flow.parallel(self.sync_build_artifact, artifacts, MAX_PARALLEL)

            for artifact_path in artifacts_paths:
                logging.info('Unpacking build artifact: {}'.format(artifact_path))
                # unpack tar.gz
                unpack_opts = ['tar', '--overwrite', '-xzf', artifact_path, '-C', target_path]
                if strip_components:
                    unpack_opts.extend(['--strip-components', str(strip_components)])
                run_process(unpack_opts, log_prefix='unpack_artifact')

        def sync_sqsh_build_artifacts(artifacts):
            def mk_symlink(dirname):
                subprocess.call([
                    'ln',
                    '-s', str(dirname),
                    os.path.join(str(target_path), os.path.basename(str(dirname)))
                ])

            def unsquash_to(artifact_path):
                subprocess.call([
                    'unsquashfs',
                    '-no-progress',
                    '-force',
                    '-dest', os.path.join(str(target_path)),
                    str(artifact_path)
                ])

            if unsquash:
                artifacts_paths = self.sync_build_artifacts(artifacts)
                for artifact_path in artifacts_paths:
                    unsquash_to(artifact_path)
            else:
                artifacts_paths = flow.parallel(self.mount_build_artifact, artifacts, MAX_PARALLEL)
                for artifact_path in artifacts_paths:
                    flow.parallel(mk_symlink, list(artifact_path.iterdir()))

        # create target directory
        target_path.mkdir(parents=True, exist_ok=True)

        tar_artifacts = filter(is_tar_artifact, resources)
        sqsh_artifacts = filter(is_sqsh_artifact, resources)
        flow.apply_async([
            (sync_tar_build_artifacts, (tar_artifacts, )),
            (sync_sqsh_build_artifacts, (sqsh_artifacts, )),
        ])

    def unpack_build_artifacts_in_overlayfs_mode(self, resources, *args, **kwargs):
        def unpack_artifact(resource):
            path = self.sync_build_artifact(resource)

            if str(path).endswith(SQUASHFS_EXT):
                return str(self.mount_build_artifact(resource))

            if str(path).endswith(TAR_GZ_EXT):
                target_dir = tempfile.mkdtemp()
                self.unpack(path, target_dir)

                return target_dir

            raise Exception('Unsupported artifact "{}"'.format(path))

        return flow.parallel(unpack_artifact, resources)

    @classmethod
    def sync_unpack_resource(cls, resource, target_path=None):
        """
        :type resource: sdk2.Resource
        :param target_path: pathlib2.Path
        """
        if target_path is None:
            target_path = pathlib2.Path.cwd()
        else:
            # create target directory
            target_path.mkdir(parents=True, exist_ok=True)

        res_path = cls.sync_build_artifacts((resource,))[0]

        if res_path.suffix == '.gz':
            unpack_opts = ['tar', '--overwrite', '-xvzf', res_path, '-C', target_path]
        else:
            unpack_opts = ['tar', '--overwrite', '-xvf', res_path, '-C', target_path]

        run_process(unpack_opts, log_prefix='unpack_res')

    @staticmethod
    def pack_into_archive(dir_path, gzip=False, archive_filename=None):
        """
        :param dir_path: path of dir to pack into archive
        :type dir_path: pathlib2.Path
        :param gzip: compress archive or not
        :type gzip: bool
        :param archive_filename: if not set, archive filename would be 'dir_path + .tar(.gz)?'
        :type archive_filename: None| pathlib2.Path
        :return: archive filename
        :rtype: pathlib2.Path
        """
        if not dir_path.is_dir():
            raise Exception('{} is not a dir'.format(dir_path))

        if archive_filename is None:
            if gzip:
                archive_filename = '{}.tar.gz'.format(dir_path)
            else:
                archive_filename = '{}.tar'.format(dir_path)

        args = '-cvf'
        if gzip:
            args = '-zcvf'

        run_process(['tar', '-C {}'.format(dir_path), args, archive_filename, './'], log_prefix='archive')

        return pathlib2.Path(archive_filename)

    def save_resources(self, resources, target_dir):
        """
        Saves resources to specified directory
        :param resources: Iterable of resource descriptor instances
        :param target_dir: Path to the target directory (will be created if not exist).
        """
        target_dir.mkdir(parents=True, exist_ok=True)

        for resource_path in self.sync_build_artifacts(resources):
            logging.info('Copy resource {} to {}'.format(resource_path, target_dir))
            shutil.copy(str(resource_path), str(target_dir))

    @staticmethod
    def get_resource(
            resource_type,
            state=(ctr.State.READY, ctr.State.NOT_READY),
            **attrs):
        """
        Поиск зарегистрированного ресурса, по дефолту не BROKEN (в том числе и NOT_READY, если не найдется READY).
        :param resource_type: идентификатор ресурса
        :param state: значение state ресурса
        :param attrs: аттрибуты ресурса
        """
        resources = list(resource_type.find(
            state=state,
            attrs=attrs
        ).order(-sdk2.Resource.id).limit(10))

        for resource in resources:
            if resource.state == ctr.State.READY:
                return resource

        return resources[0] if resources else None

    def get_dev_ready_resources(self, attrs_list, owner, repo, **common_attrs):
        """
        Возвращает список готовых ресурсов для dev ветки указанного репозитория;
        ресурсы имеют одинаковый аттрибут tree_hash.
        Для каждого набора атрибутов из списка attrs_list:
        * должно найтись не менее одного ресурса
        * возвращен будет только один ресурс
        :param attrs_list: список значений аттрибутов ресурсов
        :type attrs_list: list of dict
        :param owner: владелец репозитория
        :type owner: str
        :param repo: имя репозитория
        :type repo: str
        """

        # Находим tree hash для последних
        # 5 ресурсов (по аналогии с self._github.get_latest_merge_commits_tree_hashes - по дефолту находит столько же)
        attrs = attrs_list[0].copy()
        attrs.update(common_attrs, owner=owner, repo=repo, ref='dev')

        project_tree_hashes = map(lambda x: x.project_tree_hash, SANDBOX_CI_ARTIFACT.find(
            state=(ctr.State.READY,),
            attrs=attrs
        ).order(-sdk2.Resource.id).limit(5))

        if len(project_tree_hashes) == 0:
            logging.info('Unable to find tree hashes for resources with attrs: {}'.format(attrs))

        for project_tree_hash in project_tree_hashes:
            resources = []
            for attrs in attrs_list:
                full_attrs = attrs.copy()
                full_attrs.update(common_attrs)
                resource = self.get_resource(
                    resource_type=SANDBOX_CI_ARTIFACT,
                    state=(ctr.State.READY,),
                    project_tree_hash=project_tree_hash,
                    **full_attrs
                )
                if resource is None:
                    break

                resources.append(resource)

            if len(attrs_list) == len(resources):
                return resources

        logging.info('Unable to get any resources for tree hashes: {}'.format(project_tree_hashes))

    def get_not_ready_resources(self, limit=20):
        return list(sdk2.Resource.find(
            task=self.task,
            state=ctr.State.NOT_READY,
            type=(SANDBOX_CI_ARTIFACT, REPORT_TEMPLATES_PACKAGE, REPORT_STATIC_PACKAGE)
        ).limit(limit))

    def remove_not_ready_resources(self):
        """
        NOT_READY ресурсы упавшей таски подходят под условие "реюзабельности" (поиск по tree-hash),
        чтобы при клонировании таски "не готовые" ресурсы не могли попасть в мета-таску будем их удаялять.
        see https://st.yandex-team.ru/FEI-5393
        """
        task_resources_ids = [res.id for res in self.get_not_ready_resources()]
        logging.debug('Resources that will deleted: {}'.format(task_resources_ids))
        if task_resources_ids:
            self.task.server.batch.resources.delete = task_resources_ids

        return None

    @classmethod
    def unpack(cls, path, target_dir='.'):
        """
        :param path: путь к архиву для распаковки
        :type path: sdk2.Path
        :param path: директория, в которую небходимо распаковать архив
        :type path: str
        """
        run_process(['tar', '-xzf', path, '-C', target_dir], log_prefix='unpack')

    def get_last_artifact_resource_with_fallback(self, resource_type, attrs, fallbackAttrs=None, created=None):
        """
        looking for passed resource type with fallback to SANDBOX_CI_ARTIFACT.
        :param resource_type: Resource type class.
        :param attrs: resource attrs.
        :param fallbackAttrs: resource fallback attrs, None by default.
        :param created: filter by resource creation, None by default.
        """
        res = self.get_last_artifact_resource(
            resource_type=resource_type,
            created=created,
            **attrs
        )

        if not res:
            if fallbackAttrs:
                attrs = fallbackAttrs

            res = self.get_last_artifact_resource(
                resource_type=SANDBOX_CI_ARTIFACT,
                created=created,
                **attrs
            )

        return res

    def get_infrastats_resource(self, attrs, created=None):
        """
        looking for INFRASTATS_REPORT resource.
        :param attrs: resource attrs.
        :param created: filter by resource creation, None by default.
        """
        return self.get_last_artifact_resource(
            resource_type=InfrastatsReport,
            created=created,
            **attrs
        )
