# -*- coding: utf-8 -*-

import logging
import re
import six

from sandbox.common.types import task as ctt
from sandbox.common.errors import TaskFailure

from sandbox.projects.sandbox_ci.utils.env import snake_case

COMMON_SEMAPHORE_PREFIX = 'selenium'
AG_SEMAPHORE_PREFIX = 'selenium_autogenerated'
BROWSER_REGEX = re.compile(r'^([^(]+)\(([^)]+)\)$')


class BrowsersConfigManager(object):
    def __init__(self, task):
        self.task = task

    @staticmethod
    def parse_browser(browser):
        match = BROWSER_REGEX.search(browser)

        return {'id': match.group(1) if match else browser, 'semaphore': match.group(2) if match else browser}

    @staticmethod
    def format_sessions_per_browser_cli_option(browser_id, sessions_cnt):
        kebab_case = lambda name: re.sub(r'-?([0-9]+)', '-\\1', name)

        if not sessions_cnt:
            return ''

        return '--browsers-{}-sessions-per-browser {}'.format(kebab_case(browser_id), sessions_cnt)

    def format_sessions_per_browser_env_options(self, prefix='hermione'):
        env = {}

        for browser in self.get_browsers_to_run():
            if not browser['sessions']:
                continue

            env.update({'{}_browsers_{}_sessions_per_browser'.format(prefix, snake_case(browser['name'])): str(browser['sessions'])})

        return env

    def get_semaphores(self):
        required_semaphores = self.__get_required_semaphores()

        if not required_semaphores:
            logging.debug('Required semaphores are not found')
            return None

        use_limited_grid = self.task.config.get_deep_value(['tests', 'limited_grid'], False)

        semaphores = []
        for browser, weight in required_semaphores.items():
            if not weight:
                continue

            common_semaphore_name = '{}_{}'.format(COMMON_SEMAPHORE_PREFIX, browser)
            semaphores.append(self.__get_common_semaphore(common_semaphore_name, weight))

            if use_limited_grid:
                ag_semaphore_name = '{}_{}'.format(AG_SEMAPHORE_PREFIX, browser)
                semaphores.append(self.__get_ag_semaphore(ag_semaphore_name, common_semaphore_name, weight))

        return filter(lambda s: s is not None, semaphores)

    def __get_required_semaphores(self):
        available_semaphores = self.task.project_conf.get('tests', {}).get('semaphores', [])
        semaphores = {}
        unknown_semaphores = []
        for browser in self.get_browsers_to_run():
            semaphore = browser['semaphore']
            if semaphore in available_semaphores:
                semaphores[semaphore] = semaphores.get(semaphore, 0) + browser['sessions']
            else:
                unknown_semaphores.append(semaphore)

        if len(unknown_semaphores) > 0:
            self.task.info.report_info('Unknown semaphores', unknown_semaphores, lambda s: s)

        return semaphores

    def __get_common_semaphore(self, common_semaphore_name, weight):
        logging.debug('Using semaphore for selenium grid: {}'.format(common_semaphore_name))

        return ctt.Semaphores.Acquire(name=common_semaphore_name, weight=weight)

    def __get_ag_semaphore(self, ag_semaphore_name, common_semaphore_name, weight):
        # процент от максимально доступного
        grid_capacity = float(self.task.config.get_deep_value(['tests', 'limited_grid_capacity'], 1))
        if grid_capacity == 1:
            return

        common_semaphore_capacity = self.__get_semaphore_by_name(common_semaphore_name)['capacity']

        # искусственным образом ограничиваем капасити нового семафора,
        # чтобы не выедать полностью персистентный
        current_available_capacity = int(common_semaphore_capacity * grid_capacity)

        # мы не можем сгенерировать семафор, у которого капасити меньше, чем нужно задаче,
        # иначе она зависнет навсегда
        minimum_capacity = max(current_available_capacity, weight)

        # На всякий случай так же проверяем, что не пытаемся запросить больше чем вообще доступно всем
        ag_semaphore_capacity = min(minimum_capacity, common_semaphore_capacity)

        logging.debug('Creating semaphore {} with capacity {}'.format(ag_semaphore_name, ag_semaphore_capacity))

        return ctt.Semaphores.Acquire(name=ag_semaphore_name, weight=weight, capacity=ag_semaphore_capacity)

    def __get_semaphore_by_name(self, name):
        logging.debug('Search semaphore by name {}'.format(name))
        response = self.task.server.semaphore[{'name': name}, : 1]

        if not response:
            return None

        semaphore = next(iter(response['items']), None)
        logging.debug('Found semaphore {}'.format(semaphore))

        return semaphore

    def get_browsers_to_run(self):
        env_with_suffix = lambda suffix: '{}_{}'.format(self.task.tool.upper(), suffix)
        get_env_with_suffix = lambda suffix: self.task.environ.get(env_with_suffix(suffix), None)
        split_by_comma = lambda str: str.split(',') if str else []

        browsers_to_run = split_by_comma(get_env_with_suffix('BROWSERS'))
        should_run = lambda name: name in browsers_to_run if browsers_to_run else True

        browsers_to_skip = split_by_comma(get_env_with_suffix('SKIP_BROWSERS'))
        should_skip = lambda name: name in browsers_to_skip

        browsers = []
        for platform in self.__get_platforms():
            conf = self.get_semaphores_for_platform(platform)

            for bro, sessions in six.iteritems(conf):
                res = self.parse_browser(bro)
                name = res['id']
                semaphore = res['semaphore']

                if should_run(name) and not should_skip(name):
                    browsers.append({'name': name, 'semaphore': semaphore, 'sessions': sessions})

        known_browsers = [bro['name'] for bro in browsers]
        unknown_browsers = [bro for bro in browsers_to_run if bro not in known_browsers]
        if unknown_browsers:
            raise TaskFailure('Unknown or skipped browsers specified in {}: {}'.format(env_with_suffix('BROWSERS'), unknown_browsers))

        return browsers

    def get_semaphores_for_platform(self, platform):
        return self.task.project_conf.get('tests', {}).get(self.task.tool, {}).get('browsers', {}).get(platform, {})

    def format_external_config(self, tests):
        external_config = {}
        if not tests:
            return external_config

        for platform in self.__get_platforms():
            current_semaphores = self.task.config.get_deep_value(['tests', self.task.tool, 'browsers', platform], {})

            for browser, sessions in six.iteritems(current_semaphores):
                parsed_bro = self.parse_browser(browser)
                browser_id = parsed_bro['id']
                tests_count_for_browser = len(filter(lambda t: t['browserId'] == browser_id, tests))

                config_key = '*.tests.{}.browsers.{}.{}'.format(self.task.tool, platform, browser)
                if tests_count_for_browser > 0:
                    external_config[config_key] = min(tests_count_for_browser, sessions)
                else:
                    external_config[config_key] = 0

        return external_config

    def __get_platforms(self):
        platforms = []

        if hasattr(self.task.Parameters, 'platform') and self.task.Parameters.platform:
            platforms.append(self.task.Parameters.platform)

        if hasattr(self.task.Parameters, 'platforms'):
            platforms += self.task.Parameters.platforms

        return list(set(platforms))

    def get_platform_browsers(self, platform):
        platform_browsers = []

        for browser in self.task.project_conf.get('tests', {}).get(self.task.tool, {}).get('browsers', {}).get(platform, {}).keys():
            platform_browsers.append(self.parse_browser(browser)['id'])

        return platform_browsers
