# -*- coding: utf-8 -*-


class CacheProfilerManager(object):
    """Профайлер переиспользования кэша для задач."""

    def __init__(self, task):
        """
        :param task: инстанс задачи
        :type task: sandbox.projects.sandbox_ci.task.BaseTask
        """
        self.task = task

        if not task.Context.profiler_cache:
            task.Context.profiler_cache = []

    def register_action(self, tag, action_type, description=None, reused=True):
        """
        Регистрирует действие с кэшем

        Пример:
            self.register_action('same_task', 'task', 'reuse same task', False)

        :param tag: уникальное имя действия
        :type tag: str
        :param action_type: тип переиспользуемой сущности
        :type action_type: str
        :param description: дополнительное описание действия
        :type description: str
        :param reused: переиспользование или нет
        :type reused: bool
        :rtype: None
        """
        self.task.Context.profiler_cache.append({
            'tag': tag,
            'type': action_type,
            'reused': reused,
            'description': description,
        })

    def hit(self, tag, action_type, description=None):
        """
        Регистрирует действие переиспользования кэша

        Пример:
            self.hit('same_task', 'task')
            self.hit('SANDBOX_CI_LINTERS', 'task', 'reuse same task')

        :param tag: уникальное имя действия
        :type tag: str
        :param action_type: тип переиспользуемой сущности
        :type action_type: str
        :param description: дополнительное описание действия
        :type description: str
        :rtype: None
        """
        self.register_action(tag, action_type, description, reused=True)

    def miss(self, tag, action_type, description=None):
        """
        Регистрирует отсутствие переиспользования кэша

        Пример:
            self.miss('same_task', 'task')
            self.miss('SANDBOX_CI_LINTERS', 'task', 'reuse same task')

        :param tag: уникальное имя действия
        :type tag: str
        :param action_type: тип переиспользуемой сущности
        :type action_type: str
        :param description: дополнительное описание действия
        :type description: str
        :rtype: None
        """
        self.register_action(tag, action_type, description, reused=False)

    def get_actions(self):
        """
        Возвращает все действия переиспользования кэша в текущей задаче, включая переиспользование её дочерних задач

        :rtype: dict
        """
        cache_hit = []
        cache_miss = []

        # Дочерние задачи сами переиспользуют себе подобные задачи, т.е. информацию можно получить только у них
        same_tasks = self._get_subtasks()

        for task in same_tasks:
            task_type = task['type']
            task_reused = task['reused']

            self.hit(task_type, 'task') if task_reused else self.miss(task_type, 'task')

        cache_actions = self.task.Context.profiler_cache  # type: list

        # Маршрутизируем записи про переиспользование кэша в два списка по признаку переиспользования
        for action in cache_actions:
            item = action.copy()

            item.pop('reused')

            cache_hit.append(item) if action['reused'] else cache_miss.append(item)

        return {'cache_hit': cache_hit, 'cache_miss': cache_miss}

    def _get_subtasks(self):
        """
        Возвращает все дочерние задачи в формате: type + reused

        :rtype: list
        """
        subtasks = list(self.task.find().limit(0))

        return map(lambda task: {'type': str(task.type), 'reused': bool(task.Context.reused_same_task)}, subtasks)
