# -*- coding: utf-8 -*-

import copy
import yaml
import six
from functools import reduce
from sandbox import common
from sandbox.common.errors import TaskError


def deep_merge(base, ext):
    """
    :param base: dict
    :param ext: dict
    :rtype: dict
    """
    if ext is None:
        ext = {}

    if not isinstance(ext, dict):
        return ext
    if not isinstance(base, dict):
        return copy.deepcopy(ext)

    merge_result = copy.deepcopy(base)
    for k, v in six.iteritems(ext):
        if isinstance(merge_result.get(k), dict):
            merge_result[k] = deep_merge(merge_result[k], v)
        else:
            merge_result[k] = copy.deepcopy(v)
    return merge_result


def parse_external_conf(external_config):
    if not external_config:
        return None

    parsed_config = {}
    for key in external_config:
        parsed_config = deep_merge(parsed_config, get_config_prop(key.split('.'), external_config[key]))

    return parsed_config


def get_config_prop(key, value):
    head, tail = key[0], key[1:]
    if not tail:
        return {head: yaml.load(value)}

    return {head: get_config_prop(tail, value)}


def deep_get(_dict, keys, default=None):
    """
    :param _dict: словарь
    :type _dict: dict
    :param keys: путь
    :type keys: list of str
    :param default: значение по умолчанию
    :type default:
    """
    for key in keys:
        if isinstance(_dict, dict):
            _dict = _dict.get(key, default)
        else:
            return default

    return _dict


class ConfigManager(object):
    def __init__(self, task):
        self.task = task

    @staticmethod
    @common.utils.singleton
    def get_properties(section='sandbox-ci'):
        try:
            # genisys configuration: sandbox-ci section
            from library.config import query

            return query(
                section=section,
                subsection=None,
                evaluate=True,  # evaluate variables inside yaml values
                as_dict=True,
            )
        except KeyError:
            raise TaskError('Could not fetch config for "{section}" from Genisys, check out {url}/{section}'.format(
                section=section,
                url='https://genisys.yandex-team.ru/sections',
            ))

    @staticmethod
    def get_project_conf(conf, opts):
        """
        Собирает из общей конфигурации конфигурацию для проекта.

        Параметры вычисляются со всех уровней, и переопределяют друг друга в порядке:
            - Уровень *
            - Проектный уровень * (fiji, web4 и тд)
            - Контекстный уровень (dev, pull-request, release)

        Конфиг в genisys https://genisys.yandex-team.ru/rules/sandbox-ci/sandbox_clients
        :param conf: dict
        :param opts: dict
            :param project_name: str
            :param build_context: str
            :param external_config: str
        :rtype: dict
        """
        base_conf = conf.get('*', {})
        project_conf = conf.get(opts.get('project_name'), {})
        external_config = parse_external_conf(opts.get('external_config'))

        merged_conf = reduce(deep_merge, [base_conf, project_conf, external_config])
        build_context = opts.get('build_context')

        if build_context:
            return deep_merge(
                merged_conf.get('*', {}),
                merged_conf.get(build_context, {}),
            )

        return merged_conf.get('*', {})

    def is_enabled(self, section, sub_section):
        """
        Получает информацию о том включена ли опция в конфиге проекта.

        в событии on_save получаем конфиг проекта, а после можем получить конкретные директивы.
        В том случае, если параметр не существует метод вернёт False
        Пример конфига Genisys:
        ...
        deploy:
           yappy:
              enabled: true

        Пример использования: is_enabled('deploy', 'yappy') -> True:

        :param section: секция в конфиге
        :type section: str
        :param sub_section: подсекция
        :type sub_section: str
        :rtype bool
        """

        value = self.get_deep_value(keys=[section, sub_section, 'enabled'], default=False)
        return bool(value)

    def get_deep_value(self, keys, default=None):
        """
        :param keys: путь в конфиге
        :type keys: list of str
        :param default: дефолтное значение
        :type default:
        """
        conf = self.task.project_conf
        return deep_get(conf, keys=keys, default=default)
