# -*- coding: utf-8 -*-

from sandbox import sdk2
import os
import logging
import hashlib
import contextlib

from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.projects.sandbox_ci.utils.context import Overlayfs
from sandbox.projects.sandbox_ci.utils.process import run_process
from sandbox.projects.sandbox_ci.utils.squashfs import mksquashfs
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_BOWER_BUNDLE, SANDBOX_CI_NODE_MODULES_BUNDLE, SANDBOX_CI_BOWER_IMAGE, SANDBOX_CI_NODE_MODULES_IMAGE, SANDBOX_CI_ARTIFACT

NPM_DEPS_FILES = ('package-lock.json', 'npm-shrinkwrap.json', 'package.json')
BOWER_DEPS_FILES = ('bower.json',)


class DependenciesManager(object):
    """
    Manager that helps to install dependencies using Sandbox resources to store cache.

    Works with npm and bower out of the box. Also supports other package managers
    (see self.install_deps() method description).
    """

    def __init__(self, task, project_name):
        """
        Dependencies manager constructor

        :param task: Sandbox task instance.
        :param project_name: Project name string.
        """
        self.task = task
        self.project_name = project_name

    @property
    def use_overlayfs(self):
        return getattr(self.task, 'use_overlayfs')

    @property
    def node_modules_version(self):
        """
        Binary version of binary node modules for the current node in PATH.

        :return: Version string.
        """
        return run_process(
            ['node', '-p', 'process.versions.modules'],
            outs_to_pipe=True,
        ).communicate()[0].strip()

    def install_deps_in_overlayfs_mode(self, deps_files, target_archive, lifecycle_step, resource_type, **kwargs):
        project_deps_files = [self.task.project_dir / deps_file for deps_file in deps_files]

        attrs = dict(
            project=self.project_name,
            container=self.task.Parameters._container.id,
            **kwargs
        )

        if self.task.Parameters.reuse_dependencies_cache:
            attrs.update(checksum=files_sha1(*project_deps_files))
            res = self.task.artifacts.get_last_artifact_resource(resource_type, **attrs)
            if res:
                logging.info('Resource of type {} was found, syncing: {}'.format(resource_type, res.id))
                self.task.Context.reused_resources.append(res.id)
                self.task.cache_profiler.hit(lifecycle_step, 'resource')

                return str(sdk2.ResourceData(res).mount())
            else:
                logging.info('Resource of type {} was not found, running lifecycle step: {}'.format(resource_type, lifecycle_step))
        else:
            logging.info('No cache mode, running lifecycle step: {}'.format(lifecycle_step))

        self.task.cache_profiler.miss(lifecycle_step, 'resource')

        with Overlayfs(lower_dirs=self.task.project_dir) as (upper_dir, mount_point):
            self.task.lifecycle(lifecycle_step, work_dir=mount_point)
            mksquashfs(input=upper_dir, output=str(self.task.path(target_archive)), work_dir=self.task.working_path())

            attrs.update(checksum=files_sha1(*project_deps_files))
            self.task.artifacts.create_ready_artifact_resource(target_archive, resource_type, **attrs)

            return upper_dir

    def install_deps(self, deps_files, target_dir, lifecycle_step, resource_type, root_dir=sdk2.Path('.'),
                     target_archive=None, **kwargs):
        """
        Generic method to install dependencies using cache if exist, and creating cache if not.

        :param deps_files: Iterable of dependency files, relative to root_dir.
        :param target_dir: Target directory that dependencies will be installed into, relative to root_dir.
        :param lifecycle_step: Lifecycle step to run to install dependencies if cache does not exist.
        :param resource_type: Resource class to create to store dependencies cache.
        :param root_dir: Root directory to compute paths from, absolute or relative to self.path().
        :param target_archive: Target archive file to pack dependencies into, relative to self.path().
        :param kwargs: Additional cache resource attributes.
        :return: Cache resource descriptor instance.
        """
        if not root_dir.is_absolute():
            root_dir = self.task.working_path(root_dir)

        # resolve target_dir
        target_dir_abs = root_dir / target_dir
        target_dir_rel = target_dir_abs.relative_to(self.task.working_path())
        target_dir_rel_len = len(target_dir_rel.parts)

        # resolve deps_files
        project_deps_files = [root_dir / deps_file for deps_file in deps_files]
        logging.info('Installing dependencies into {} using files: {}'.format(
            target_dir_rel, ', '.join(map(str, project_deps_files)),
        ))

        # get checksum from deps_files
        checksum = files_sha1(*project_deps_files)
        logging.info('Dependency files checksum: {}'.format(checksum))

        # resource attributes
        attrs = dict(
            project=self.project_name,
            checksum=checksum,
            # path=str(target_archive or target_dir_rel),  # conflicts with `path` argument in Resource constructor
            target_dir=str(target_dir_rel),
            container=self.task.Parameters._container.id,
            **kwargs
        )
        attrs_log = ', '.join(['{}={}'.format(k, v) for (k, v) in attrs.items()])

        logging.info('Trying to find resource bundle of type {} with attributes: {}'.format(
            resource_type.name, attrs_log,
        ))

        if self.task.Parameters.reuse_dependencies_cache:
            res = self.task.artifacts.get_last_artifact_resource(resource_type, **attrs)
            if res:
                logging.info('Resource of type {} found with id, syncing: {}'.format(resource_type.name, res.id))
                res_path = sdk2.ResourceData(res).path

                self.task.Context.reused_resources.append(res.id)
                self.task.cache_profiler.hit(lifecycle_step, 'resource')

                logging.info('Cleanup target path: {}'.format(target_dir_abs))
                # delete target directory
                run_process(['rm', '-rf', target_dir_abs])

                if target_archive:
                    # create target directory
                    target_dir_abs.mkdir(mode=0o777, parents=True, exist_ok=True)

                    logging.info('Unpacking resource contents from {} into {}'.format(res_path, target_dir_abs))
                    # unpack tar.gz
                    run_process([
                        'tar', '--strip-components={}'.format(target_dir_rel_len), '-xzf', res_path,
                        '-C', target_dir_abs,
                    ])
                else:
                    logging.info('Copying resource contents from {} into {}'.format(res_path, target_dir_abs))
                    run_process(['cp', '-al', res_path, target_dir_abs])
                return res
            else:
                logging.info('Resource of type {} not found, running lifecycle step: {}'.format(
                    resource_type.name, lifecycle_step,
                ))
        else:
            logging.info('No cache mode, running lifecycle step: {}'.format(lifecycle_step))

        self.task.cache_profiler.miss(lifecycle_step, 'resource')
        self.task.lifecycle(lifecycle_step)

        if not target_dir_abs.exists():
            raise Exception('Target directory {} not found after lifecycle step: {}'.format(
                target_dir_abs, lifecycle_step,
            ))

        resource_path = target_dir
        if target_archive:
            resource_path = target_archive
            resource_dir = resource_path.parent

            if resource_dir and not resource_dir.exists():
                logging.info('Creating directory for tar.gz: {}'.format(resource_dir))
                resource_dir.mkdir(mode=0o777, parents=True, exist_ok=True)

            logging.info('Packing {} into {}'.format(target_dir, self.task.path(target_archive)))
            run_process(
                ['tar', '-zcf', self.task.path(target_archive), target_dir_rel],
                work_dir=self.task.working_path(),
            )

        installed_checksum = files_sha1(*project_deps_files)
        if checksum != installed_checksum:
            changed_file = project_deps_files[0]
            logging.debug('{file} was changed during installation: {old_checksum} -> {new_checksum}'.format(
                file=changed_file,
                old_checksum=checksum,
                new_checksum=installed_checksum,
            ))
            attrs.update(checksum=installed_checksum)
            attrs_log = ', '.join(['{}={}'.format(k, v) for (k, v) in attrs.items()])

        # save target_dir content as a resource with checksum as one of attributes
        logging.info('Creating resource of type {} from {} with attributes: {}'.format(
            resource_type.name, resource_path, attrs_log,
        ))
        new_res = self.task.artifacts.create_artifact_resource(resource_path, resource_type, **attrs)
        sdk2.ResourceData(new_res).ready()
        return new_res

    @in_case_of('use_overlayfs', 'npm_install_in_overlayfs_mode')
    def npm_install(self, target_dir=sdk2.Path('node_modules'), target_archive=sdk2.Path('node_modules.tar.gz'),
                    lifecycle_step='npm_install'):
        """
        Install npm dependencies using cache if exists, otherwise create cache. With respect to binary modules version.

        :param target_dir: Target directory that dependencies will be installed into.
        :param target_archive: Target archive file to pack dependencies into.
        :param lifecycle_step: Dependency installation step in life cycle.
        :return: Cache resource descriptor instance.
        """
        os.environ.setdefault('NPM_CONFIG_REGISTRY', 'https://npm.yandex-team.ru')
        target_dir_dir = target_dir.parent
        deps_files = [target_dir_dir / f for f in NPM_DEPS_FILES]

        return self.install_deps(
            deps_files,
            root_dir=sdk2.Path(self.project_name),
            target_dir=target_dir,
            target_archive=target_archive,
            lifecycle_step=lifecycle_step,
            resource_type=SANDBOX_CI_NODE_MODULES_BUNDLE,
            node_modules_version=self.node_modules_version,
        )

    def npm_install_in_overlayfs_mode(self, target_archive=sdk2.Path('node_modules.squashfs'), lifecycle_step='npm_install'):
        os.environ.setdefault('NPM_CONFIG_REGISTRY', 'https://npm.yandex-team.ru')

        return self.install_deps_in_overlayfs_mode(
            NPM_DEPS_FILES,
            target_archive,
            lifecycle_step,
            resource_type=SANDBOX_CI_NODE_MODULES_IMAGE,
            node_modules_version=self.node_modules_version,
        )

    @in_case_of('use_overlayfs', 'bower_install_in_overlayfs_mode')
    def bower_install(self, target_dir=sdk2.Path('libs'), target_archive=sdk2.Path('libs.tar.gz')):
        """
        Install bower dependencies using cache if exists, and creating cache if not.

        :param target_dir: Target directory that dependencies will be installed into.
        :param target_archive: Target archive file to pack dependencies into.
        :return: Cache resource descriptor instance.
        """
        target_dir_dir = target_dir.parent
        deps_files = [target_dir_dir / f for f in BOWER_DEPS_FILES]

        return self.install_deps(
            deps_files,
            root_dir=sdk2.Path(self.project_name),
            target_dir=target_dir,
            target_archive=target_archive,
            lifecycle_step='bower_install',
            resource_type=SANDBOX_CI_BOWER_BUNDLE,
        )

    def bower_install_in_overlayfs_mode(self, target_archive=sdk2.Path('libs.squashfs'), lifecycle_step='bower_install'):
        return self.install_deps_in_overlayfs_mode(
            BOWER_DEPS_FILES,
            target_archive,
            lifecycle_step,
            resource_type=SANDBOX_CI_BOWER_IMAGE,
        )

    def blockstat_install(self, dict_name='blockstat', target_file=sdk2.Path('blockstat.dict.json'),
                          resource_type=SANDBOX_CI_ARTIFACT):
        """
        Install blockstat dictionary using cache if exists.

        :param dict_name: Name of statface dictionary
        :type dict_name: str
        :param target_file: Target file that will be installed into, relative to root_dir.
        :type target_file: sdk2.Path
        :param resource_type: Resource class to store dependencies cache.
        :type resource_type: sdk2.Resource
        """
        res = self.task.artifacts.get_last_artifact_resource(resource_type, dict=dict_name, format='pairs')
        if not res:
            logging.info('Resource of type {} not found'.format(resource_type.name))
            return

        logging.info('Resource of type {} found with id, syncing: {}'.format(
            resource_type.name, res.id,
        ))
        res_path = sdk2.ResourceData(res).path

        self.task.Context.reused_resources.append(res.id)
        self.task.cache_profiler.hit(dict_name, 'resource')

        root_dir = sdk2.Path(self.project_name)

        if not root_dir.is_absolute():
            root_dir = self.task.working_path(root_dir)

        target_path_abs = root_dir / target_file

        logging.info('Copying resource contents from {} into {}'.format(
            res_path, target_path_abs,
        ))

        run_process(['cp', res_path, target_path_abs])
        run_process(['chmod', '+w', target_path_abs])


def file_sha1_update(path, hasher):
    """
    Updates hasher with the single file.

    :param path: Path to the file to update hasher with.
    :param hasher: Hash object from hashlib
    """
    with contextlib.closing(open(str(path), 'rb')) as binary_file:
        while True:
            data = binary_file.read(4096)
            if not data:
                break
            hasher.update(data)


def file_sha1(path):
    """
    Compute sha1 checksum for the single file.

    :param path: Path to the file to compute sha1 checksum for.
    :return: sha1 string
    """
    hasher = hashlib.sha1()
    file_sha1_update(path, hasher)
    return hasher.hexdigest()


def files_sha1(*args):
    """
    Compute sha1 checksum for the multiple files (only existing ones), order matters

    :param args: Paths to files.
    :return: sha1 string
    :raise IOError in case when no files exist.
    """
    existing_paths = filter(lambda path: path.exists(), args)
    if len(existing_paths) == 0:
        raise IOError('No files found: {}'.format(', '.join(map(str, args))))

    hasher = hashlib.sha1()
    for path in existing_paths:
        file_sha1_update(path, hasher)

    return hasher.hexdigest()
