# -*- coding: utf-8 -*-

import logging
import re
import json

from sandbox.common.utils import Enum

class MergeQueueErrorNames(Enum):
    """ Error types to send to feedback service """
    MERGE_CONFLICT = 'MERGE_CONFLICT'
    ARC_MERGE_CONFLICT = 'ARC_MERGE_CONFLICT'
    UNSTAGED_CHANGES = 'UNSTAGED_CHANGES'
    UNSTAGED_LFS = 'UNSTAGED_LFS'
    AUTOSQUASH_CONFLICT = 'AUTOSQUASH_CONFLICT'
    TASK_EXCEPTION = 'TASK_EXCEPTION'
    FAILED_POLLING_REQUIRED_CHECKS = 'FAILED_POLLING_REQUIRED_CHECKS'
    FAILED_PROJECT_REQUIRED_CHECKS = 'FAILED_PROJECT_REQUIRED_CHECKS'
    ARCADIA_GITHUB_CONFIG_ERROR = 'ARCADIA_GITHUB_CONFIG_ERROR'
    REVIEW_REQUEST_INVALID_STATE = 'REVIEW_REQUEST_INVALID_STATE'
    REVIEW_REQUEST_INVALID_HEAD = 'REVIEW_REQUEST_INVALID_HEAD'


class MergeQueueError(object):
    name = ''
    description_title = ''
    feedback = ''
    messages = []

    def __init__(self, messages, code=None):
        self.code = code
        if isinstance(messages, list):
            self.messages = messages
        else:
            self.messages = [messages]

    @property
    def description(self):
        description = [self.description_title]
        description.extend(self.messages)

        return '<br>'.join(description)


class MergeQueueTaskException(MergeQueueError):
    name = MergeQueueErrorNames.TASK_EXCEPTION
    description_title = '<b>Unexpected task exception:</b>'
    feedback = 'Unexpected task exception, see logs for more details'

class FailedPollingRequiredChecksError(MergeQueueError):
    name = MergeQueueErrorNames.FAILED_POLLING_REQUIRED_CHECKS
    description_title = '<b>Checks failure:</b>'
    feedback = 'Failed polling required checks'

class FailedProjectRequiredChecksError(MergeQueueError):
    name = MergeQueueErrorNames.FAILED_PROJECT_REQUIRED_CHECKS
    description_title = '<b>Checks failure:</b>'
    feedback = 'Failed project required checks'

class GitMergeConflictError(MergeQueueError):
    name = MergeQueueErrorNames.MERGE_CONFLICT
    description_title = '<b>Merge conflicts:</b>'
    feedback = 'Merge conflict, see logs for more details'

class ArcMergeConflictError(MergeQueueError):
    name = MergeQueueErrorNames.ARC_MERGE_CONFLICT
    description_title = '<b>Merge conflicts:</b>'
    feedback = 'Merge conflict, see logs for more details'

class GitAutosquashConflictError(MergeQueueError):
    name = MergeQueueErrorNames.AUTOSQUASH_CONFLICT
    description_title = '<b>Autosquash conflicts:</b>'
    feedback = 'Autosquash conflict, see logs for more details'

class GitUnstagedLfsError(MergeQueueError):
    name = MergeQueueErrorNames.UNSTAGED_LFS
    description_title = '<b>Unstaged binaries:</b>'
    feedback = 'Found binary files instead of lfs pointers after rebase, see logs for more details'

class GitUnstagedChangesError(MergeQueueError):
    name = MergeQueueErrorNames.UNSTAGED_CHANGES
    description_title = '<b>Unstaged binaries:</b>'
    feedback = 'Found binary files instead of lfs pointers after rebase, see logs for more details'

class ArcadiaGithubConfigError(MergeQueueError):
    name = MergeQueueErrorNames.ARCADIA_GITHUB_CONFIG_ERROR
    description_title = '<b>Config error:</b>'
    feedback = 'Failed to find Arcadia <-> GitHub pair'

class ReviewRequestInvalidStateError(MergeQueueError):
    name = MergeQueueErrorNames.REVIEW_REQUEST_INVALID_STATE
    description_title = '<b>Review request assertions failed:</b>'

    @property
    def feedback(self):
        return self.messages[0]

class ReviewRequestInvalidHeadError(MergeQueueError):
    name = MergeQueueErrorNames.REVIEW_REQUEST_INVALID_HEAD
    description_title = '<b>Review request assertions failed:</b>'

    @property
    def feedback(self):
        return self.messages[0]

class MergeQueueErrorParser:
    stderr_matchers = [
        # Этот матчер всегда должен идти перед MERGE_CONFLICT, поскольку является
        # частным случаем ошибки мерж-конфликта.
        {
            'pattern': r'rebase -i --autosquash.*\n(CONFLICT \(.+\):.*\n[\n\S\s]*fixup!.*)$',
            'error':  GitAutosquashConflictError
        },
        {
            'pattern': r'(CONFLICT \(.+\):.*)$',
            'error': GitMergeConflictError
        },
        {
            'pattern': r'stderr: (there are some conflicts:[\s\S]*?)\nrebase',
            'error': ArcMergeConflictError
        },
        # Порядок обработки unstaged_* имеет значение, поскольку unstaged lfs является частным случаем unstaged
        # и попадает под оба матчера unstaged_*. Поэтому он должен выполняться первым.
        {
            'pattern': r'(Encountered \d+ file\(s\) that should have been pointers, but weren\'t:\n[\n\r\t\S]*\n\n)',
            'error': GitUnstagedLfsError
        },
        {
            'pattern': r'(cannot rebase: You have unstaged changes.)$',
            'error': GitUnstagedChangesError
        }
    ]

    message_matchers = [
        {
            'pattern': r'^Failed polling required checks',
            'error': FailedPollingRequiredChecksError
        },
        {
            'pattern': r'^Failed project required checks',
            'error': FailedProjectRequiredChecksError
        },
        {
            'pattern': r'^Failed to find Arcadia <-> GitHub pair',
            'error': ArcadiaGithubConfigError
        },
        {
            'pattern': r'^review request \d* head commit is "[0-9a-zA-Z]*", expected "[0-9a-zA-Z]*"',
            'error': ReviewRequestInvalidHeadError
        },
        {
            'pattern': r'^review request \d* state is "[a-zA-Z]*", expected "[a-zA-Z]*"',
            'error': ReviewRequestInvalidStateError
        }
    ]

    @staticmethod
    def parse_from_stderr(stderr):
        logging.debug('Parsing known errors from stderr')

        for matcher in MergeQueueErrorParser.stderr_matchers:
            match = re.findall(matcher['pattern'], stderr, re.MULTILINE)

            if match:
                logging.debug('Found known error in stderr with match {}'.format(match))
                error_class = matcher['error']

                return error_class(match)

    @staticmethod
    def parse_from_exception(e):
        logging.debug('Parsing known errors from exception message "{}"'.format(e.message))

        for matcher in MergeQueueErrorParser.message_matchers:
            match = re.match(matcher['pattern'], e.message)

            if match:
                logging.debug('Found known error in exception message "{}"'.format(e.message))
                error_class = matcher['error']

                return error_class(e.message)

        return MergeQueueTaskException(e.message)


# Здесь должен быть @dataclass, но в мы в Python2
class MergeQueueFormattedError:
    def __init__(self, errorCode, message, name):
        self.errorCode = errorCode
        self.message = message
        self.name = name

    @staticmethod
    def parse_merge_queue_formatted_error(error_json):
        try:
            error_data = json.loads(error_json)

            return MergeQueueFormattedError(error_data['errorCode'], error_data['message'], error_data['name'])
        except ValueError:
            raise MergeQueueTaskException(
                "merge failed: decode error: error msg format is not JSON: \"%s\"" % error_json
            )
        except KeyError as e:
            raise MergeQueueTaskException(
                "merge failed: keyError: missing required error field: \"%s\"" % e.args[0]
            )


class ArcanumMergeQueueCliError(Exception, MergeQueueError):
    def __init__(self, json_error=None):
        error_data = MergeQueueFormattedError.parse_merge_queue_formatted_error(json_error)
        message = "Arcanum merge queue cli failed: %s" % error_data.message
        Exception.__init__(self, message)
        MergeQueueError.__init__(self, message, error_data.errorCode)

        self.name = error_data.name
        self.description_title = "<b>Arcanum merge queue cli error:</b>"
        self.feedback = message
