# -*- coding: utf-8 -*-
import inspect
import os
import logging
from sandbox.common.utils import get_task_link

from sandbox.projects.sandbox_ci.utils.github import convert_sandbox_status_to_github_state, GitHubApi


def warn_about_scp_feedback():
    if not is_called_from_scp_feedback():
        logging.debug('Warning! Creating github status by direct call GitHubStatusesManager. '
                      'Use scp_feedback.py instead. Call stack: {}'.format(inspect.stack()))


def is_called_from_scp_feedback():
    """
    Метод позволяет определить, было ли вызвано создание статуса в обход модуля scp_feedback.
    Делается в рамках перехода к Arcanum checks —
    необходимо подменить везде вызов создания GH статуса напрямую из этого модуля
    """
    for frame in inspect.stack():
        if 'scp_feedback.py' in frame[1]:
            return True

    return False


class GitHubStatusesManager(object):
    def __init__(self, task):
        """
        :param task:
        :type task: sandbox.projects.sandbox_ci.task.BaseTask
        """
        self.task = task

    def create_github_status(self, owner, repo, context, sha, state, url, description):
        """
        Создаёт статус коммита на GitHub с указанными параметрами
        :param owner: владелец репозитория
        :type owner: str
        :param repo: имя репозитория
        :type repo: str
        :param context: контекст github статуса
        :type context: str
        :param sha: хеш коммита
        :type sha: str
        :param state: статус коммита
        :type state: str
        :param url: адрес, на который ведёт статус
        :type url: str
        :param description: описание статуса
        :type description: str
        :rtype: None
        """
        github_api_token = os.environ.get('GITHUB_API_TOKEN')

        if github_api_token:
            api = GitHubApi(token=github_api_token)
        else:
            # В on_enqueue нет доступка к секретам, поэтому будем делать запросы через специальный сервер, который их
            # добавит.
            # @see: SANDBOX-3882, FEI-10158
            api = GitHubApi(host='github-token-proxy.si.yandex-team.ru')

        warn_about_scp_feedback()
        return api.create_status(owner, repo, context, sha, state, url, description)

    @property
    def report_statuses_config(self):
        """
        :rtype: bool
        """
        return self.task.project_conf.get('github', {}).get('report_statuses', False)

    @property
    def should_report_statuses(self):
        """
        По параметрам таски и конфигу вычисляет, должна ли она репортить статус.
        :rtype: bool
        """
        return self.task.Parameters.report_github_statuses and self.report_statuses_config

    def report_status(self, owner, repo, context, sha, state, url, description='', force=False):
        """
        Репортит статус коммита на GitHub с указанными параметрами, если этого требует задача и включено в Genisys
        :param owner: владелец репозитория
        :type owner: str
        :param repo: имя репозитория
        :type repo: str
        :param context: контекст github статуса
        :type context: str
        :param sha: хеш коммита
        :type sha: str
        :param state: статус коммита
        :type state: str
        :param url: адрес, на который ведёт статус
        :type url: str
        :param description: описание статуса
        :type description: str
        :param force: отправить статус независимо от параметров задачи и конфигурации в genisys
        :type force: bool
        :rtype: None
        """
        commit_id = '{owner}/{repo}@{sha}'.format(owner=owner, repo=repo, sha=sha)
        fmt_attrs = dict(commit_id=commit_id, context=context, state=state, description=description)

        if not force and not self.should_report_statuses:
            warn_about_scp_feedback()
            logging.debug(u'{commit_id}: skip creating {state} status "{context}" ({description})'.format(**fmt_attrs))
            return

        logging.debug(u'{commit_id}: creating {state} status "{context}" ({description})'.format(**fmt_attrs))
        return self.create_github_status(owner, repo, context, sha, state, url, description)

    def safe_report_status(self, owner, repo, context, sha, state, url, description='', force=False):
        """
        Безопасно репортит статус коммита на GitHub с указанными параметрами, если этого требует задача и репортинг
        включён в Genisys
        :param owner: владелец репозитория
        :type owner: str
        :param repo: имя репозитория
        :type repo: str
        :param context: контекст github статуса
        :type context: str
        :param sha: хеш коммита
        :type sha: str
        :param state: статус коммита
        :type state: str
        :param url: адрес, на который ведёт статус
        :type url: str
        :param description: описание статуса
        :type description: str
        :param force: отправить статус независимо от параметров задачи и конфигурации в genisys
        :type force: bool
        :rtype: None
        """
        try:
            # Получаем коммит по SHA. Необходимо для проверки того, что коммит существует перед тем, как мы будем
            # выставлять ему статус. Подробне: FEI-6755
            GitHubApi().get_commits(owner, repo, sha)

            self.report_status(owner, repo, context, sha, state, url, description, force)
        except:
            logging.exception(u'Unable to create a GitHub status for {owner}/{repo}#{sha}'.format(
                owner=owner,
                repo=repo,
                sha=sha,
            ))

    def report_status_to_current_sha(self, context, state, url, description, safe=False, force=False):
        """
        Репортит статус на текущий коммит с указанными параметрами
        :param context: контекст github статуса
        :type context: str
        :param state: статус коммита
        :type state: str
        :param url: адрес, на который ведёт статус
        :type url: str
        :param description: описание статуса
        :type description: str
        :param safe: необходимость безопасного выставления статуса коммита
        :type safe: bool
        :rtype: None
        """
        if not self.can_report_status_to_current_sha():
            return

        status_reporter = self.safe_report_status if safe else self.report_status

        status_reporter(
            owner=self.task.Parameters.project_github_owner,
            repo=self.task.Parameters.project_github_repo,
            sha=self.task.Parameters.project_github_commit,
            context=context,
            state=state,
            url=url,
            description=description,
            force=force,
        )

    def can_report_status_to_current_sha(self):
        """
        Проверяет что все необходимые параметры для репортинга статусов проставленны
        Если не проствленны, кидает warning
        :rtype: bool
        """
        required_params = (
            'project_github_owner',
            'project_github_repo',
            'project_github_commit',
        )

        for param in required_params:
            if not getattr(self.task.Parameters, param, None):
                logging.warning(u"Can't report GitHub status, {} parameter must be defined".format(param))
                return False

        return True

    def report_self_status(self, state=None, description='', safe=False):
        """
        Репортит свой статус в GitHub с указанным описанием
        :param context: контекст github статуса
        :type state: str
        :param url: адрес, на который ведёт статус
        :type url: str
        :param description: описание статуса
        :type description: str
        :param safe: необходимость безопасного выставления статуса коммита
        :type safe: bool
        :rtype: None
        """
        self.report_status_to_current_sha(
            context=self.task.github_context,
            state=state if state else convert_sandbox_status_to_github_state(self.task.status),
            url=get_task_link(self.task.id),
            description=description,
            safe=safe,
        )

    def safe_report_self_status(self, state=None, description=''):
        """
        Безопасно репортит свой статус в GitHub с указанным описанием
        :param context: контекст github статуса
        :type state: str
        :param url: адрес, на который ведёт статус
        :type url: str
        :param description: описание статуса
        :type description: str
        :rtype: None
        """
        self.report_self_status(state, description, True)
