# -*- coding: utf-8 -*-

import logging
import shutil
import subprocess
import urllib2
import os
import re
import json

from functools import partial
from datetime import datetime, timedelta

from sandbox import sdk2

from sandbox.common.types import resource as ctr
from sandbox.common.utils import get_task_link

from sandbox.projects.sandbox_ci.utils import flow
from sandbox.projects.sandbox_ci.utils.process import run_process
from sandbox.projects.sandbox_ci.utils.context import Node


def get_last_html_report(report_type, task_id):
    return sdk2.Resource.find(task_id=task_id, attrs={'type': report_type}).first()


class HtmlReportManager(object):
    def __init__(self, task):
        self.task = task

    def publish_merged_sqlite_report(self, report_type, output_dirname, status, attrs={}, running_task_ids=[], finished_task_ids=[], output_report_type=None):
        """
        Публикует смерженный html отчёт hermione для случая, где saveFormat == sqlite. Сабтаски могут быть ещё не завершены.

        :param running_task_ids: id запущенных задач. В runtime отчёта будет искаться databaseUrls.json из последних ресурсов (связанных с этими задачами) с типом report_type.
        :param finished_task_ids: id завершившихся задач. Функция найдёт последние ресурсы из этих задач с типом report_type и подставит в отчёт их id.
        """
        try:
            plugin = self.__get_html_reporter_plugin(output_report_type or report_type, output_dirname)

            databases_paths = []

            for id in running_task_ids:
                databases_paths.append(
                    'https://proxy.sandbox.yandex-team.ru/last/SANDBOX_CI_ARTIFACT/databaseUrls.json?attrs={{"type":"{}"}}&task_id={}'.format(report_type, id)
                )

            resources = filter(bool, flow.parallel(partial(get_last_html_report, report_type), finished_task_ids))
            for res in resources:
                databases_paths.append('https://proxy.sandbox.yandex-team.ru/{}/databaseUrls.json'.format(res.id))

            self.__merge_sqlite_reports(databases_paths, output_dirname)

            plugin.make_reports(status, attrs)

            return True
        except Exception as e:
            logging.error('Cannot create merged sqlite report: %s', str(e))

            return False

    def publish_merged_html_reports(self, task_ids, report_type, output_dirname, status, output_report_type=None, tar=False, attrs={}):
        try:
            plugin = self.__get_html_reporter_plugin(output_report_type or report_type, output_dirname)
            reports_paths = filter(bool, flow.parallel(partial(self.__load_html_report, report_type, tar), task_ids))

            if not len(reports_paths):
                return

            self.__merge_html_reports(reports_paths, output_dirname)

            plugin.make_reports(status, attrs)
            return True
        except Exception as e:
            logging.error('Cannot merge html reports: %s', str(e))
            return False

    def __get_html_reporter_plugin(self, report_type, output_dirname):
        from sandbox.projects.sandbox_ci.task.test_task.plugins import HtmlReporter
        plugin = HtmlReporter(self.task)

        plugin.set_report_type(report_type)
        plugin.set_report_path(output_dirname)
        plugin.set_env()

        return plugin

    def __load_html_report(self, report_type, tar, task_id):
        html_report_resource = get_last_html_report(report_type, task_id)

        if not html_report_resource:
            return

        task = sdk2.Task.find(id=task_id, children=True).limit(1).first()
        src_path = str(task.project_path(sdk2.ResourceData(html_report_resource, fastbone=True).path))
        dest_path = '{}/{}-{}'.format(self.task.project_dir, report_type, task_id)

        if not tar:
            logging.info('Copying report from {} to {}'.format(src_path, dest_path))
            shutil.copytree(src_path, dest_path)
            subprocess.call(['chmod', '-R', '0775', dest_path])
        else:
            logging.info('Unpacking report from {} to {}'.format(src_path, dest_path))
            subprocess.call(['mkdir', '-p', dest_path])
            subprocess.call(['tar', '--overwrite', '-xzf', src_path, '-C', dest_path])

        return dest_path

    def __merge_sqlite_reports(self, databases_paths, output_dirname):
        with Node(self.task.Parameters.node_js_version):
            command_args = [
                './node_modules/.bin/hermione',
                '--config', '.{}.conf.js'.format(self.task.tool),
                'merge-reports',
                '--html-reporter-save-format', 'sqlite',
                '-d', output_dirname
            ] + databases_paths

            run_process(
                command_args,
                log_prefix='merge_sqlite_reports',
                shell=False,
                work_dir=self.task.project_dir
            )

    def __merge_html_reports(self, reports_paths, output_dirname):
        with Node(self.task.Parameters.node_js_version):
            command_args = [
                './node_modules/.bin/hermione',
                '--config', '.{}.conf.js'.format(self.task.tool),
                'merge-reports',
                '-d', output_dirname
            ] + reports_paths

            run_process(
                command_args,
                log_prefix='merge_html_reports',
                shell=False,
                work_dir=self.task.project_dir
            )

    def get_html_report_url(self, tool):
        report_res = sdk2.Resource.find(task_id=self.task.id, state=ctr.State.READY, attrs={'type': '{}-report'.format(tool)}).first()

        if report_res is None:
            return None

        report_url = '{}/index.html'.format(report_res.http_proxy)
        try:
            check_res = urllib2.urlopen(report_url)
            if check_res.code == 200:
                return report_url
        except:
            logging.info('Report with path {} does not exist'.format(report_url))

        return None

    def get_data_urls(self):
        database_urls_json = self.get_database_urls_json() or {}
        db_urls = database_urls_json.get('dbUrls') or []
        return '\n'.join(db_urls)

    def get_database_urls_json(self):
        if getattr(self.task.Parameters, 'html_reporter_use_sqlite', False):
            report_dir = '{}-report'.format(self.task.tool)
            db_urls_path = str(self.task.project_dir.joinpath(report_dir, 'databaseUrls.json'))

            try:
                with open(db_urls_path) as json_data:
                    return json.load(json_data)
            except (IOError, ValueError) as e:
                logging.warning('Cannot read {}: {}'.format(db_urls_path, str(e)))

        return None

    def format_html_report_url(self, report_url=None, browsers=None, default=None):
        if not report_url:
            return default or get_task_link(self.task.id)

        return '{}?{}'.format(report_url, '&'.join(map(lambda browser: 'browser={}'.format(browser), browsers))) if browsers else report_url

    def format_platform_html_report_url(self, report_url, platform, default=None):
        return self.format_html_report_url(report_url, self.task.browsers_config.get_platform_browsers(platform), default)

    def get_extra_items(self, parent=None):
        task = parent or self.task

        extra_items = {
            'sandbox task': 'https://sandbox.yandex-team.ru/task/{}'.format(task.id),
            'restart task': 'https://sandbox-ci.si.yandex-team.ru/clone-task?sandboxTaskId={}'.format(task.id)
        }

        if re.match(r'^pull/(\d+)$', task.Parameters.ref):
            extra_items['pull request'] = 'https://github.yandex-team.ru/{}/{}/{}'.format(
                task.Parameters.project_github_repository.project_github_owner,
                task.Parameters.project_github_repository.project_github_repo,
                task.Parameters.ref)

        infrastats_report_link = self.__get_infrastats_report_link()

        if infrastats_report_link != '':
            extra_items['infrastats'] = infrastats_report_link

        return json.dumps(extra_items)

    def set_extra_items(self, parent=None):
        if os.environ.get('HTML_REPORTER_EXTRA_ITEMS'):
            return

        os.environ['HTML_REPORTER_EXTRA_ITEMS'] = self.get_extra_items(parent)

    def __get_infrastats_report_link(self):
        if not hasattr(self.task, 'project_name'):
            return ''

        project = '{}-{}'.format(self.task.project_name, self.task.Parameters.service) \
            if hasattr(self.task.Parameters, 'service') else self.task.project_name

        tool = self.task.tool.replace('-e2e', '_e2e')

        report_resource = self.__find_recent_infrastats_report_resource({
            'status': 'SUCCESS',
            'type': '{}-{}'.format(project, tool),
            'released': 'stable'
        })

        # fallback for platform reports
        if report_resource is None and hasattr(self.task.Parameters, 'platform') and self.task.Parameters.platform:
            report_resource = self.__find_recent_infrastats_report_resource({
                'status': 'SUCCESS',
                'type': '{}-{}-{}'.format(project, tool, self.task.Parameters.platform)
            })

        if report_resource is None:
            return ''

        logging.debug("Found resource with infrastats report: #%r", report_resource.id)

        return '{}/index.html'.format(report_resource.http_proxy)

    def __find_recent_infrastats_report_resource(self, attrs):
        """
        Returns infrastats report created in the last 24 hours.
        """
        return self.task.artifacts.get_infrastats_resource(
            attrs=attrs,
            created='{}..{}'.format((datetime.now() - timedelta(days=1)).isoformat(), datetime.now().isoformat())
        )
