# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox.common.utils import get_task_link, get_resource_link


class InfoManager(object):
    task_link_template = '<a href="{url}">{type}#{id}</a>'

    def __init__(self, task):
        self.task = task

    def report_info(self, title, items, formatter):
        if not items:
            return

        lines = ['<li>{}</li>'.format(formatter(item)) for item in items]

        self.task.set_info('{title}:<br/><ul>{lines}</ul>'.format(
            title=title,
            lines=''.join(lines)
        ), do_escape=False)

    @staticmethod
    def task_link_html(task):
        return InfoManager.task_link_template.format(
            id=task.id,
            url=get_task_link(task.id),
            type=task.type
        )

    @staticmethod
    def resource_link(resource):
        return '<a href="{url}">{type}#{id}</a>'.format(
            id=resource.id,
            url=get_resource_link(resource.id),
            type=resource.type
        )

    def report_failed_tasks(self, failed_tasks_ids):
        if not failed_tasks_ids:
            return

        failed_tasks = sdk2.Task.find(id=failed_tasks_ids, children=True).limit(len(failed_tasks_ids))
        self.report_info('Failed tasks', failed_tasks, self.task_link_html)

    def report_not_finished_tasks(self, not_finished_tasks_ids):
        if not not_finished_tasks_ids:
            return

        not_finished_tasks = sdk2.Task.find(id=not_finished_tasks_ids, children=True).limit(len(not_finished_tasks_ids))
        self.report_info('Not finished tasks', not_finished_tasks, self.task_link_html)

    def report_reused_task(self):
        if not self.task.Context.reused_same_task:
            return

        task_id = self.task.Context.same_task
        task_link = self.task_link_template.format(
            id=task_id,
            url=get_task_link(task_id),
            type=self.task.type,
        )

        self.task.set_info('Reused task: {}'.format(task_link), do_escape=False)

    def report_reused_resources(self, reused_resources_ids):
        if not reused_resources_ids:
            return

        reused_resources = sdk2.Resource.find(id=reused_resources_ids).limit(len(reused_resources_ids))
        self.report_info('Reused resources', reused_resources, self.resource_link)

    def report_all(self):
        self.report_reused_task()
        self.report_failed_tasks(self.task.Context.failed_tasks)
        self.report_not_finished_tasks(self.task.Context.not_finished_tasks)
        self.report_reused_resources(self.task.Context.reused_resources)
