# -*- coding: utf-8 -*-

import logging
import json

from sandbox import sdk2

from sandbox.projects.sandbox_ci.utils import flow
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT


class JsonReport(object):
    @staticmethod
    def read(report_path):
        """
        :param report_path: path to json-report file
        :type report_path: str
        :return: report
        :rtype: JsonReport
        """
        with open(str(report_path)) as f:
            return JsonReport(json.load(f))

    @staticmethod
    def load(report_resource):
        """
        :param report_resource: json-report resource
        :type report_resource: sandbox.sdk2.Resource
        :return: report
        :rtype: JsonReport
        """
        return JsonReport.read(report_resource.task.path(sdk2.ResourceData(report_resource).path))

    def __init__(self, data=None):
        """
        :param data: json-report data
        :type data: dict
        """
        self._data = data if data else {}

    def data(self):
        """
        :return: json-report data
        :rtype: dict
        """
        return self._data

    def is_empty(self):
        """
        :return: checks whether report is empty
        :rtype: bool
        """
        return len(self._data) == 0

    def failed_tests(self):
        """
        :return: failed tests
        :rtype: list of dict
        """
        return filter(lambda t: t['status'] == 'fail', self._data.values())

    def has_failed_tests(self, browsers):
        """
        :param browsers: list of browser ids
        :type browser: list of str
        return: checks whether report has fails in the specified browsers
        :rtype: bool
        """
        for failed_test in self.failed_tests():
            if failed_test['browserId'] in browsers:
                return True

        return False

    def update(self, other):
        """
        Updates report with data from `other` report.

        :param other: json-report
        :type other: JsonReport
        :return: self
        :rtype: JsonReport
        """
        self._data.update(other.data())

        return self

    def merge(self, other):
        """
        Updates report with data from `other` report but preserves retries info.

        :param other: json-report
        :type other: JsonReport
        :return: self
        :rtype: JsonReport
        """
        for key, other_test in other.items():
            if key in self._data:
                test = self._data[key]
                retries = test.get('retries', []) + other_test.get('retries', [])
                if retries:
                    other_test = dict(other_test, retries=retries)

            self._data[key] = other_test

        return self

    def save(self, path):
        """
        :param path: path
        :type path: str
        """
        with open(str(path), 'w') as f:
            f.write(json.dumps(self._data, ensure_ascii=False).encode('utf8') if self._data else u'{}')

    def items(self):
        return self._data.items()

    def __setitem__(self, key, item):
        self._data[key] = item

    def __getitem__(self, key):
        return self._data[key]

    def __contains__(self, key):
        return key in self._data


class JsonReportManager(object):
    def __init__(self, task):
        """
        :param task: task
        """
        self.task = task

    @property
    def __resource_type(self):
        return 'json-reporter'

    def create(self, data=None):
        """
        :param data: report data
        :type data: dict
        :return: report
        :rtype: JsonReport
        """
        return JsonReport(data)

    def read(self, report_path):
        """
        :param report_path: path to json-report file
        :type report_path: str
        :return: report
        :rtype: JsonReport
        """
        return JsonReport.read(report_path)

    def safe_read(self, report_path):
        """
        :param report_path: path to json-report file
        :type report_path: str
        :return: report
        :rtype: JsonReport
        """
        try:
            return self.read(report_path)
        except Exception as e:
            logging.debug('Cannot read json report from path {}: {}'.format(report_path, e))
            return JsonReport({})

    def load(self, report_resource):
        """
        :param report_resource: json-report resource
        :type report_resource: sandbox.sdk2.Resource
        :return: report
        :rtype: JsonReport
        """
        return JsonReport.load(report_resource)

    def safe_load(self, report_resource):
        """
        :param report_resource: json-report resource
        :type report_resource: sandbox.sdk2.Resource
        :return: report
        :rtype: JsonReport
        """
        try:
            return self.load(report_resource)
        except Exception as e:
            logging.debug('Cannot load json report resource {}: {}'.format(report_resource, e))
            return JsonReport({})

    def publish_merged_json_reports(self, task_ids, report_path, status, tags, attrs):
        """
        :param task_ids: task id
        :type task_ids: list of int
        :param report_path: report path
        :type report_path: str
        :param status: resource status
        :type status: sandbox.common.types.task.Status
        :param tags: task tags
        :type tags: list of str
        :param attrs: resource attributes
        :type attrs: dict
        :return: merged json-report resource
        :rtype: sandbox.sdk2.Resource
        """
        try:
            merged_report = JsonReport({})
            for res in flow.parallel(self.get_task_report_resource, task_ids):
                if res:
                    merged_report.merge(self.load(res))

            if merged_report.is_empty():
                return

            merged_report_file_path = self.task.working_path(report_path)
            merged_report.save(merged_report_file_path)

            return self.task.artifacts.create_report(
                resource_path=merged_report_file_path,
                task_id=self.task.id,
                type=self.__resource_type,
                status=status,
                tags=tags,
                **attrs
            )
        except Exception as e:
            logging.exception('Cannot merge json reports: %s', e)

    def get_task_report_resource(self, task_id):
        """
        :param task_id: task id
        :type task_id: int
        :return: json-report resource
        :rtype: sandbox.sdk2.Resource
        """
        return self.task.artifacts.get_last_artifact_resource(
            resource_type=SANDBOX_CI_ARTIFACT,
            task_id=task_id,
            type=self.__resource_type,
        )
