# -*- coding: utf-8 -*-
import logging
import time
import six

from sandbox.sandboxsdk import process

SAFE_TIMEOUT_MARGIN = 10 * 60


class LifecycleManager(object):
    def __init__(self, task, steps, project_dir=None, variables=None):
        self.task = task
        self.steps = steps
        self.project_dir = project_dir or task.path()
        self.vars = variables if variables is not None else dict(
            working_dir=str(self.task.working_path()),
            scripts_dir=str(self.task.scripts.path()),
            project_dir=str(self.task.project_dir),
            project_name=self.task.project_name,
        )

    def __call__(self, step, **kwargs):
        """
        Запуск шага жизненного цикла задачи, если тот существует.

        :param step: Название шага
        :type step: str
        :param kwargs:
        :return:
        """
        step_command = self.steps.get(step, 'true')

        if step_command == 'true':
            logging.warning('"{}" is not set in lifecycle steps. Instead "true" will be used.'.format(step))

        params = dict(work_dir=self.project_dir, shell=True, log_prefix=step)
        if isinstance(step_command, dict):
            params.update({key: step_command[key] for key in step_command if key != 'cmd'})
            step_command = step_command['cmd']

        if isinstance(step_command, six.string_types):
            step_command = self.format_command(step_command)
        elif isinstance(step_command, (list, tuple)):
            step_command = map(self.format_command, step_command)

        params.update(kwargs)
        params.update(work_dir=self.format_command(params['work_dir']))
        if not params.get('timeout'):
            params['timeout'] = self.safe_timeout()

        return process.run_process(step_command, **params)

    def safe_timeout(self):
        if not isinstance(self.task.Context.execution_started_at, float):
            return None

        running_time = int(time.time() - self.task.Context.execution_started_at)

        return self.task.Parameters.kill_timeout - running_time - SAFE_TIMEOUT_MARGIN

    def has_step(self, step):
        """
        Проверяет наличие шага жизненного цикла задачи.

        :param step: Название шага
        :type step: str
        :rtype: bool
        """
        return bool(self.steps.get(step))

    def format_command(self, cmd):
        """
        Форматирует переданную команду, используя метод `str.format()` и `self.vars`.

        :param cmd: Команда для форматирования
        :type cmd: str
        :rtype: str
        """
        # Приводим к utf-8 кодировке только unicode
        if isinstance(cmd, six.text_type):
            cmd = cmd.encode('utf-8')

        # Подготавливаем словарь переменных к использованию в форматтере
        variables = self.prepare_vars(self.vars)

        return str(cmd).format(**variables)

    def update_vars(self, **kwargs):
        """
        Устанавливает или обновляет переменные в `self.vars`, используемые для форматирование команд шага.

        :param kwargs: Необходимые переменные
        :return: self
        """
        # Приводим строковые значения параметров к utf-8, чтобы избежать ошибок при появлении кириллицы в командах.
        self.vars.update(self.prepare_vars(kwargs))

        return self

    def prepare_vars(self, variables):
        """
        Подготавливает словарь переменных к использованию.

        Sandbox возвращает значения параметров, определённых с типом String, в unicode. При этом параметры отличных
        типов от String возвращаются с соответствующими типами.

        :param variables: Словарь переменных
        :type variables: dict
        :rtype: dict
        """
        for (k, v) in variables.items():
            # unicode встречается в значениях параметров (в ENV его использовать нельзя – Sandbox падает)
            if isinstance(v, six.text_type):
                variables[k] = v.encode('utf-8')

        return variables
